/*
** filterlib.c - written in milano by vesely on 19 oct 2001
** for courier-mta global filters (with variations from
** original libfilter.c)
**
** This is a modular software, not object oriented:
** compile with:
**       "-DFILTER_NAME=blah_blah"
*/
/*
** Copyright (c) 1999-2025 Alessandro Vesely
This file is part of avfilter

avfilter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

avfilter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License version 3
along with avfilter.  If not, see <http://www.gnu.org/licenses/>.

*/
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#if !AVFILTER_DEBUG && !NDEBUG
#define NDEBUG
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <syslog.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <signal.h>
#include <time.h>

#include <avfiledefs.h>

#if !defined(PATH_MAX)
#define PATH_MAX 256
#endif

#if defined(FILTER_NAME)
#define STRING2(P) #P
#define STRING(P) STRING2(P)
#define THE_FILTER STRING(FILTER_NAME)
#else
#define THE_FILTER "filterlib"
#endif

#include "log.h"
#include "vb_fgets.h"
#include "filterlib.h"
#include <assert.h>

static volatile int
	signal_timed_out = 0,
	signal_break = 0,
	signal_hangup = 0;
static int live_children = 0;

#if 0
static char const filter_pass[] = "X-checked-" THE_FILTER;
#endif

typedef struct ctl_fname_chain
{
	 struct ctl_fname_chain *next;
	 char fname[1]; // actually large as needed
} ctl_fname_chain;

struct filter_lib_struct
{
	void *parm;

	char const *resp;
	int in, out;
	char *data_fname, *write_fname;
	FILE *data_fp, *write_fp;
	ctl_fname_chain *cfc;
	char *argv0;

	fl_callback filter_fn;
	sigset_t blockmask, allowset;

	int ctl_count;
	unsigned int all_mode:4;
	unsigned int verbose:4;
	unsigned int testing:4;
	unsigned int batch_test:4;
	unsigned int write_file:2;
	unsigned int alllocal:1;
};

/* ----- sig handlers ----- */

static int sig_verbose = 0;
static void child_reaper(int sig)
{
	int status;
	pid_t child;

	(void)sig;

	int save_errno = errno;
	while ((child = waitpid(-1, &status, WNOHANG)) > 0)
	{
		--live_children;

#if !defined(NDEBUG)
		if (sig_verbose >= 8)
		{
			char buf[80];
			unsigned s = snprintf(buf, sizeof buf,
				THE_FILTER ": %d exited, %d remaining\n",
				(int)child, live_children);
			if (s >= sizeof buf)
			{
				buf[sizeof buf - 1] = '\n';
				s = sizeof buf;
			}
			write(2, buf, s);
		}
#endif
	}
	errno = save_errno;
}

int fl_log_no_pid;
static inline int my_getpid(void)
{
	return fl_log_no_pid? 0: (int)getpid();
}

static void sig_catcher(int sig)
{
#if !defined(NDEBUG)
	if (sig_verbose >= 1)
	{
		char buf[80];
		unsigned s = snprintf(buf, sizeof buf,
			THE_FILTER "[%d]: received signal %d: keep=%d\n",
			my_getpid(), sig, fl_keep_running());
		if (s >= sizeof buf)
		{
			buf[sizeof buf - 1] = '\n';
			s = sizeof buf;
		}
		write(2, buf, s);
	}
#endif
	switch(sig)
	{
		case SIGALRM:
			signal_timed_out = 1;
			break;

		case SIGHUP:
		case SIGUSR1:
		case SIGUSR2:
			signal_hangup = sig;
			break;

		case SIGPIPE:
		case SIGINT:
		case SIGQUIT:
		case SIGTERM:
			signal_break = sig;
			break;
		default:
			break;
	}
}

void fl_alarm(unsigned seconds)
{
#if !defined(NDEBUG)
	/* check the signal handler has been set up */
	struct sigaction oact;
	if (sigaction(SIGALRM, NULL, &oact) != 0 ||
		oact.sa_handler != &sig_catcher)
			fprintf(stderr, "SIGALRM not correct!!\n");
#endif
	signal_timed_out = 0;
	alarm(seconds);
}

int fl_keep_running(void)
{
	return signal_timed_out == 0 && signal_break == 0;
}

/* ----- ctl_fname_chain handling ----- */

static ctl_fname_chain* cfc_shift(ctl_fname_chain **cfc)
{
	ctl_fname_chain *s = *cfc;
	if (s)
		*cfc = s->next;
	return s; // freed by caller
}

static int cfc_unshift(ctl_fname_chain **cfc, char const *fname, size_t len)
{
	ctl_fname_chain *const u = (ctl_fname_chain*)
		malloc(len + sizeof(ctl_fname_chain));
	if (u)
	{
		u->next = *cfc;
		memcpy(u->fname, fname, len);
		u->fname[len] = 0;
		*cfc = u;
		return 0;
	}
	return 1;
}

/* ----- fl_* aux functions ----- */
static const stdlog_t fl_report = &stdlog;


void *fl_get_parm(fl_parm*fl)
{
	assert(fl);
	return fl->parm;
}

void fl_set_parm(fl_parm *fl, void* parm)
{
	assert(fl);
	fl->parm = parm;
}

void fl_set_verbose(fl_parm*fl, int verbose)
{
	assert(fl);
	fl->verbose = sig_verbose = verbose;
}

int fl_get_verbose(fl_parm*fl)
{
	assert(fl);
	return fl->verbose;
}

void fl_pass_message(fl_parm*fl, char const *resp)
/*
* see courier/cdfilters.C for meaning of responses:
* each line should start with three digits; if the first digit is
* '0', '4', or '5', execution of filters is stopped and the response
* is given to the remote client --replacing the first '0' in the first
* line with '2', or rejecting the message.
* 2xx responses are not parsed. However, we log them.
*/
{
	assert(fl);
	fl->resp = resp;
}

char const* fl_get_filename(fl_parm*fl)
{
	assert(fl);
	return fl->data_fname;
}

FILE* fl_get_file(fl_parm*fl)
{
	assert(fl);
	return fl->data_fp;
}

FILE *fl_get_write_file(fl_parm *fl)
{
	assert(fl);

	if (fl->write_fp == NULL)
	{
		if (fl->write_fname == NULL)
		{
			assert(fl->data_fname);
			
			char buf[PATH_MAX];
			int sz = snprintf(buf, sizeof buf, "%s" THE_FILTER "%d",
				fl->data_fname, my_getpid());
			if (sz < 0 || (unsigned)sz >= sizeof buf ||
				(fl->write_fname = strdup(buf)) == NULL)
			{
				fl_report(LOG_CRIT, "writename %s", strerror(errno));
				return NULL;
			}
		}

		if ((fl->write_fp = fopen(fl->write_fname, "w")) == NULL)
		{
			fl_report(LOG_CRIT, "cannot fopen %s: %s",
				fl->write_fname, strerror(errno));
			free(fl->write_fname);
			fl->write_fname = NULL;
		}
	}

	return fl->write_fp;
}

fl_test_mode fl_get_test_mode(fl_parm* fl)
{
	assert(fl);
	return fl->testing ? fl->batch_test ?
		fl_batch_test : fl->alllocal? 1 /* FIXME */: fl_testing : fl_no_test;
}

#if 0
void fl_report(int severity, char const* fmt, ...)
{
	char const *logmsg;
	switch (severity) // see liblog/logger.c
	{
		case LOG_EMERG:
		case LOG_ALERT:
			logmsg = "ALERT";
			break;

		case LOG_CRIT:
			logmsg = "CRIT";
			break;

		case LOG_ERR:
		default:
			logmsg = "ERR";
			break;

		case LOG_WARNING:
			logmsg = "WARN";
			break;

		case LOG_NOTICE:
		case LOG_INFO:
			logmsg = "INFO";
			break;

		case LOG_DEBUG:
			logmsg = "DEBUG";
			break;
	}

	fprintf(stderr, "%s:" THE_FILTER "[%d]:", logmsg, my_getpid());
	va_list ap;
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fputc('\n', stderr);
}

static void print_alert(char const*fname, char const* msg, int ctl, int ctltot)
{
	fprintf(stderr, "ALERT:"
		THE_FILTER "[%d]: error %s ctl file %d/%d (%s): %s\n",
		my_getpid(), msg,
		ctl, ctltot, fname, strerror(errno));
}
#endif

/* ----- ctlfile and rcpt callback ----- */


/* read single record from ctlfile, via callback */
static int
read_ctlfile(fl_parm *fl, char const *chs, int (*cb)(char *, void*), void* arg)
{
	int ctl = 0, rtc = 0;
	ctl_fname_chain *cfc = fl->cfc;

	while (cfc && rtc == 0)
	{
		char const *irtc = NULL;
		FILE *fp = fopen(cfc->fname, "r");
		++ctl;
		if (fp)
		{
			char buf[4096];
			
			while (fgets(buf, sizeof buf - 1, fp))
			{
				char *ends = strchr(buf, '\n');
				if (ends)
					*ends = 0;
				else
				{
					int c;
					while ((c = fgetc(fp)) != EOF && c != '\n')
						continue;
					buf[sizeof buf -1] = 0;
				}

				if (buf[0] &&
					strchr(chs, buf[0]) != NULL &&
					(rtc = (*cb)(&buf[0], arg)) != 0)
						break;
			}
			if (ferror(fp))
				irtc = "reading";

			fclose(fp);
		}
		else
			irtc = "opening";

		if (irtc)
			fl_report(LOG_ALERT,
				"error %s ctlfile %s (%d/%d): %s",
				irtc, cfc->fname, ctl, fl->ctl_count, strerror(errno));

		cfc = cfc->next;
	}

	return rtc;
}

/* get (auth) sender callback */
static int my_strdup(char *s, void*arg)
{
	char **rtc = (char**)arg;
	assert(rtc && *rtc == NULL);
	*rtc = strdup(s + 1);
	return 1;
}

/* get sender, rtc freed by caller */
char *fl_get_sender(fl_parm *fl)
{
	assert(fl);

	if (fl_get_test_mode(fl) == fl_testing)
		return strdup("test@test.test"); // dummy ctlfile given (the ids)

	char *rtc = NULL;

	read_ctlfile(fl, "s", &my_strdup, &rtc);
	return rtc;
}

/* get sender, rtc freed by caller */
char *fl_get_authsender(fl_parm *fl)
{
	assert(fl);

	if (fl_get_test_mode(fl) == fl_testing)
		return NULL; // dummy ctlfile given (the ids)

	char *rtc = NULL;

	read_ctlfile(fl, "i", &my_strdup, &rtc);
	return rtc;
}

int rcpt_iterate(fl_parm *fl, fl_rcpt_callback fn)
// 0 = success, -1 error, other value interrupted by callback
{
	assert(fl);
	assert(fn);

	if (fl_get_test_mode(fl) == fl_testing) // dummy ctl file
	{
		int rtc = (*fn)(fl->parm, "1@test.example");
		if (rtc == 0) rtc = (*fn)(fl->parm, "2@test.example");
		if (rtc == 0) rtc = (*fn)(fl->parm, "3@test.example");
		return rtc;
	}

	var_buf vb;
	if (vb_init(&vb))
		return -1;

	int rtc = 0;

	ctl_fname_chain *cfc = fl->cfc;
	while (cfc)
	{
		FILE *fp = fopen(cfc->fname, "r");
		if (fp)
		{
			char *p;
			while ((p = vb_fgets(&vb, 0, fp)) != NULL)
			{
				if (*p == 'r')
				{
					char *eol = p + strlen(p) - 1;
					int ch = 0;

					while (eol > p && isspace(ch = *(unsigned char*)eol))
						*eol-- = 0;

					if ((rtc = (*fn)(fl->parm, p + 1)) != 0)
						break;
				}
			}
			fclose(fp);
		}
		else
		{
			fl_report(LOG_CRIT, "cannot open %s: %s",
				cfc->fname, strerror(errno));
			rtc = -1;
		}

		if (rtc)
			break;

		cfc = cfc->next;
	}
	vb_clean(&vb);

	return rtc;
}

static int count_recipients(FILE *fp, char** msgid)
{
	int count = 0;
	char buf[2048];

	if (fseek(fp, 0, SEEK_SET) != 0)
		return -1;

	while (fgets(buf, sizeof(buf), fp))
	{
		char *ends = strchr(buf, '\n');
		if (ends)
			*ends = 0;
		else
		{
			int c;
			while ((c = fgetc(fp)) != EOF && c != '\n')
				continue;
		}

		if (buf[0] == 'r')
			++count;
		else if (buf[0] == 'M' && msgid && *msgid == NULL)
			*msgid = strdup(&buf[1]);
	}
	if (ferror(fp) || fseek(fp, 0, SEEK_END) != 0)
		return -1;

	return count;
}

int fl_drop_message(fl_parm*fl)
{
	int ctl = 0, rtc = 0;
	time_t tt;
	char *msgid = NULL;

	if (fl->verbose >= 7)
	{
		fprintf(stderr,
			THE_FILTER "[%d]: about to drop msg with %d ctl file(s)\n",
			my_getpid(), fl->ctl_count);
	}

	if (fl_get_test_mode(fl) == fl_testing)
		return 0; // dummy ctlfile given (the ids)

	time(&tt);

	while (fl->cfc)
	{
		int irtc = 0, goterrno = 0;
		ctl_fname_chain *cfc = cfc_shift(&fl->cfc);
		FILE *fp;

		++ctl;
		errno = 0;
		fp = fopen(cfc->fname, "r+");
		if (fp)
		{
			int i;
			int const count = count_recipients(fp, &msgid);

			for (i = 0; i < count && !ferror(fp); ++i)
				fprintf(fp, "I%d R 250 Dropped.\nS%d %ld\n",
					i, i, (long)tt);
			fprintf(fp, "C%ld\n", (long)tt);

			if (count < 0 || ferror(fp))
			{
				irtc = 1;
				if (goterrno == 0)
					goterrno = errno;
			}
			fclose(fp);
			if (fl->verbose >= 7)
			{
				fprintf(stderr,
					THE_FILTER
					"[%d]: dropped %d/%d recipient(s) time=%ld on ctl file %s\n",
					my_getpid(), i, count, (long)tt, cfc->fname);
			}
		}
		else
		{
			irtc = 1;
			goterrno = errno;
		}

		if (irtc)
		{
			rtc = irtc;
			fprintf(stderr, "ALERT:"
				THE_FILTER "[%d]: error on ctl file %d/%d (%s): %s\n",
				my_getpid(), ctl, fl->ctl_count, cfc->fname, strerror(goterrno));
		}
		else if (fl->verbose >= 1)
		/*
		** main logging function
		** (given as error as the rest of refused messages)
		*/
		{
			fl_report(LOG_INFO,
				"drop msg,id=%s", msgid? msgid: "");
		}

		free(cfc);
	}

	free(msgid);
	return rtc;
}

/* ----- core filter functions ----- */
typedef struct process_fname
{
	char buf[1024];
	unsigned count;
	int found, empty;
	char *data_fname;
} process_fname;

static void process_read_fname(process_fname *prof, fl_parm* fl)
{
	if (prof->buf[prof->count] != '\n')
		++prof->count;
	else if (prof->count == 0) /* empty line ends filenames */
		prof->empty = 1;
	else
	{
		prof->buf[prof->count] = 0;

#if !defined(NDEBUG)
		if (fl->verbose >= 8)
			fprintf(stderr, THE_FILTER
				"[%d]: piped fname[%d]: %s (len=%u)\n",
				my_getpid(), prof->found, prof->buf, prof->count);
#endif

		if (++prof->found == 1) /* first line */
			prof->data_fname = strdup(prof->buf);
		else if (prof->count > 1 ||
			prof->buf[0] != ' ') /* discard dummy placeholders */
		{
			if (cfc_unshift(&fl->cfc, prof->buf, prof->count))
				fprintf(stderr, "ALERT:"
					THE_FILTER "[%d]: malloc on filename #%d\n",
						my_getpid(), prof->found);
		}
		prof->count = 0;
	}
}

static int read_fname(fl_parm* fl)
/*
** This reads data and ctl file names. A file name must be either
** an absolute path or relative to the current directory (as usual.)
** When running as an installed filter, lf_init does chdir to
** COURIER_HOME, which is what was configured as @prefix@: so this
** is consistent whith courierfilter.html which states that path
** names may be relative to that directory. (However, it looks as
** if "LOCALSTATEDIR/tmp" is always used as a an absolute path.)
*/
{
	int const fd = fl->in;
	process_fname prof;
	prof.data_fname = NULL;
	prof.count = 0;
	prof.found = prof.empty = 0;
	int rtc = 0;

#if !defined(NDEBUG)
	if (fl->verbose >= 8)
		fprintf(stderr, THE_FILTER "[%d]: reading fd %d\n",
			my_getpid(), fd);
#endif

	fl_alarm(30);
	unsigned p = read(fd, prof.buf, sizeof prof.buf);
	if (p != (unsigned)(-1))
		for (prof.count = 0; prof.count < p;)
		{
			unsigned const len = prof.count + 1;
			process_read_fname(&prof, fl);
			if (prof.empty)
				break;

			if (prof.count == 0)
			{
				p -= len;
				memmove(&prof.buf[0], &prof.buf[len], p);
			}
		}

	if (fl->verbose >= 8 && prof.empty == 0)
		fl_report(LOG_DEBUG, "reading %d names not completed by first call",
			prof.found);

	// read any remaining info, one byte at a time to find the empty line
	while (prof.count < sizeof prof.buf && prof.empty == 0 && fl_keep_running())
	{
		p = read(fd, &prof.buf[prof.count], 1);
		if (p == (unsigned)(-1))
		{
			switch (errno)
			{
				case EAGAIN:
				case EINTR:
					continue;
				default:
					break;
			}
			fl_report(LOG_ALERT, "cannot read fname pipe: %s", strerror(errno));
			break;
		}
		else if (p != 1)
		{
			fl_report(LOG_ALERT, "Unexpected %s (%d) on fname pipe",
				p == 0 ? "EOF" : "rtc from read", p);
			break;
		}

		process_read_fname(&prof, fl);
	}

	alarm(0);
	fl->ctl_count = prof.found - 1;
	if (!fl_keep_running() || prof.found < 2 ||
		prof.empty == 0 || prof.count >= sizeof prof.buf ||
		prof.data_fname == NULL)
	{
		rtc = 1;
		if (!fl_keep_running())
			fl_report(LOG_ALERT, "reading fname pipe terminated prematurely");
		if (prof.found != 2 || prof.empty == 0 || prof.data_fname == NULL)
			fl_report(LOG_ALERT,
				"found%s %d file name(s)%s%s",
				prof.found < 2 ? " only" : "", prof.found,
				prof.empty ? "" : " no empty line",
				prof.data_fname == NULL ? " malloc failure" : "");
		if (prof.count >= sizeof prof.buf)
			fl_report(LOG_ALERT, "Buffer overflow reading fname pipe");
		free(prof.data_fname);
		prof.data_fname = NULL;
	}

	fl->data_fname = prof.data_fname;
	return rtc;
}

static void fl_reset_signal(void)
{
	struct sigaction act;
	memset(&act, 0, sizeof act);

	sigemptyset(&act.sa_mask);
	act.sa_handler = SIG_DFL;

	sigaction(SIGALRM, &act, NULL);
	sigaction(SIGPIPE, &act, NULL);
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGTERM, &act, NULL);

	act.sa_handler = SIG_IGN;
	sigaction(SIGCHLD, &act, NULL);
	sigaction(SIGHUP, &act, NULL);
	sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGUSR2, &act, NULL);
}

static void fl_init_signal(fl_parm *fl)
{
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);

	act.sa_flags = SA_NOCLDSTOP | SA_RESTART;
	act.sa_handler = child_reaper;
	sigaction(SIGCHLD, &act, NULL);

	act.sa_flags = SA_RESTART;
	act.sa_handler = sig_catcher;

	sigaction(SIGALRM, &act, NULL);
	sigaction(SIGPIPE, &act, NULL);
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGTERM, &act, NULL);
	sigaction(SIGHUP, &act, NULL);
	sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGUSR2, &act, NULL);

	sigemptyset(&fl->blockmask);
	sigaddset(&fl->blockmask, SIGALRM);
	sigaddset(&fl->blockmask, SIGPIPE);
	sigaddset(&fl->blockmask, SIGINT);
	sigaddset(&fl->blockmask, SIGTERM);
	sigaddset(&fl->blockmask, SIGHUP);
	sigaddset(&fl->blockmask, SIGUSR1);
	sigaddset(&fl->blockmask, SIGUSR2);
}

void fl_rollback_write(fl_parm *fl)
{
	assert(fl);

	if (fl->write_fp)
	{
		fclose(fl->write_fp);
		fl->write_fp = NULL;
	}

	if (fl->write_fname)
	{
		unlink(fl->write_fname);
		free(fl->write_fname);
		fl->write_fname = NULL;
	}	
}

int fl_commit_write(fl_parm *fl)
/*
* close and rename the temporary output file, fail on errors
*/
{
	assert(fl);

	if (fl->data_fp)
	{
		fclose(fl->data_fp);
		fl->data_fp = NULL;
	}

	if (fl->write_fp && fl->write_fname)
	{
		int fail = ferror(fl->write_fp);
		if (fail)
			fl_report(LOG_CRIT, "error writing %s: %s",
				fl->write_fname, strerror(errno));
		
		if (fclose(fl->write_fp))
		{
			fl_report(LOG_CRIT, "error closing %s: %s",
				fl->write_fname, strerror(errno));
			fail = 1;
		}
		fl->write_fp = NULL;

		if (fail == 0 && rename(fl->write_fname, fl->data_fname))
		{
			fl_report(LOG_CRIT, "error renaming %s to %s: %s",
				fl->data_fname, fl->data_fname, strerror(errno));
			fail = 1;
		}

		if (fail)
			fl_rollback_write(fl);

		return fail;
	}

	return 0;
}

static void do_the_real_work(fl_parm* fl)
{
	char const *resp = NULL;

	if ((fl->data_fp = fopen(fl->data_fname, "r")) != NULL)
	{
		/* alarm will kill after resetting */
		fl_reset_signal();
		if (fl_get_test_mode(fl) != fl_testing)
			alarm(900); // 15 minutes (way too much)

		fl->resp = NULL;
		if (fl->filter_fn)
			(*fl->filter_fn)(fl);

		alarm(0);

		resp = fl->resp;

		if (fl->verbose && !fl_keep_running())
			fl_report(LOG_ERR, "interrupted");

		if (fl->data_fp) // possibly closed in already
		{
			fclose(fl->data_fp);
			fl->data_fp = NULL;
		}

		/*
		* commit writing if not done already
		*/
		if (fl->write_fp && fl->write_fname)
		{
			if (fl_commit_write(fl) == 0)
				resp = fl->resp;
		}
		else
			resp = fl->resp;
	}
	else
	{
		fl_report(LOG_CRIT, "cannot open %s: %s\n",
			fl->data_fname, strerror(errno));
	}

	// some cleanup here (should be in the calling function)
	while (fl->cfc)
		free(cfc_shift(&fl->cfc));
	free(fl->data_fname);
	fl->data_fname = NULL;

	/*
	** give response --Courier (cdfilters.C) closes the connection when it
	** gets the last line of the response, and proceeds accordingly.
	*/
	if (resp == NULL)
	{
		resp = "432 Mail filter temporarily unavailable.\n";
		if (fl->resp == NULL)
		{
			fprintf(stderr, "ERR:"
				THE_FILTER "[%d]: response was NULL!!\n",
				my_getpid());
			fl->resp = resp;
		}
	}

	int const fd_out = fl->out;
	if (fd_out >= 0)
	{
		unsigned w = 0, l = strlen(resp);
		while (w < l && fl_keep_running())
		{
			unsigned p = write(fd_out, resp + w, l - w);
			if (p == (unsigned)(-1))
			{
				switch (errno)
				{
					case EAGAIN:
					case EINTR:
						continue;
					default:
						break;
				}
				fprintf(stderr, "ALERT:"
					THE_FILTER "[%d]: unable to write resp: %s\n",
					my_getpid(), strerror(errno));
				break;
			}
			w += p;
		}
	}
}

static pid_t my_fork(char const *what)
{
	pid_t pid;
	int retry = 2;

	while ((pid = fork()) < 0 && --retry >= 0)
		sleep(1);

	if (pid > 0) /* parent */
		++live_children;
	else if (pid < 0)
		fl_report(LOG_CRIT,
			"cannot fork child for %s: %s",
				what, strerror(errno));

	return pid;
}

#if !defined(NDEBUG)
static void fl_break(void)
{
	fputs("execution resumed\n", stderr);
}
#endif

static void fl_runchild(fl_parm* fl)
{
	pid_t pid = my_fork("filtering");

	if (pid == 0) /* child */
	{
		if (fl->verbose >= 8)
			fprintf(stderr, THE_FILTER "[%d]: started child\n",
				my_getpid());

#if !defined(NDEBUG)
		if (fl_get_test_mode(fl) == fl_testing)
		{
			char *debug = getenv("DEBUG_FILTER");
			if (debug)
			{
				int nsec = atoi(debug);
				if (nsec <= 0) nsec = 30;

				/* hallo! */
				fprintf(stderr, "DEBUG_FILTER: %s\n"
					"now you have %d secs to\n"
					"%% gdb %s %d\n"
					"(gdb) break fl_break\n"
					"Breakpoint 1 at 0xxx: file %s, line %d.\n"
					"(gdb) cont\n",
						debug, nsec, fl->argv0, (int)getpid(),
						__FILE__, __LINE__ -40);
				nsec = sleep(nsec);
				fl_break();
			}
			else if (fl->verbose >= 2)
				fprintf(stderr,
					"set DEBUG_FILTER to debug the child in gdb\n");
		}
#endif

		int rtc = 0;
		if (read_fname(fl))
			rtc = 1;
		else
			do_the_real_work(fl);

		exit(rtc);
	}
}

static int my_lf_accept(int listensock, sigset_t *allowset)
/*
** copied from courier/filters/libfilter/libfilter.c
** changed: different return code for shutting down (0 instead of -1)
** changed: use pselect if available; assume signals are blocked on entry
*/
{
	struct sockaddr_un ssun;
	fd_set fd0;
	int fd;
	socklen_t sunlen;

	if (listensock <= 0)
		return 0;

	for (;;)
	{
		FD_ZERO(&fd0);
		FD_SET(0, &fd0);
		FD_SET(listensock, &fd0);

#if HAVE_PSELECT
		if (pselect(listensock+1, &fd0, 0, 0, 0, allowset) < 0)
#else
		sigset_t blockset;
		sigprocmask(SIG_SETMASK, allowset, &blockset);
		int rtc = select(listensock+1, &fd0, 0, 0, 0);
		sigprocmask(SIG_SETMASK, &blockset, NULL);
		if (rtc < 0)
#endif
		{
			if (errno == EAGAIN || errno == EINTR)
				return -1;

			fl_report(LOG_CRIT,
#if HAVE_PSELECT
				"p"
#endif
				"select() error: %s", strerror(errno));
			continue; // changed for 3.3, was return -1 in any case
		}

		if (FD_ISSET(0, &fd0))
		{
			char buf[16];

			if (read(0, buf, sizeof(buf)) <= 0)
				return 0; /* 0 is Shutting down (cannot be accepted socket) */
		}

		if (!FD_ISSET(listensock, &fd0))
			continue;

		sunlen = sizeof ssun;
		if ((fd = accept(listensock, (struct sockaddr*)&ssun, &sunlen)) < 0)
		{
			if (errno == EAGAIN || errno == EINTR)
				return -1; // changed for 3.0, was continue in any case;

			fl_report(LOG_CRIT, "accept() error: %s", strerror(errno));
			continue;
		}

		fcntl(fd, F_SETFL, 0);  /* Take out of NDELAY mode */
		break;
	}

	return fd;
}

static int fl_runtest(fl_parm* fl, int argc, char *argv[])
{
	int rtc = 0;
	int mypipe[2], i;
	if (pipe(mypipe) == 0)
	{
		FILE *fp = fdopen(mypipe[1], "w");

		if (fp)
		{
			for (i = 0; i < argc; ++i)
			{
				fl->in = mypipe[0];
				fl->out = 1; // stdout
				fl_runchild(fl);
#if !defined(NDEBUG)
				if (fl->verbose >= 8)
					fprintf(stderr, THE_FILTER "[%d]: Running %d child(ren)\n",
						my_getpid(), live_children);
#endif
				fprintf(fp, "%s\nthe ids\n\n", argv[i]);
				fflush(fp);
			}
			fclose(fp);
		}
		else
		{
			close(mypipe[1]);
			rtc = 2;
		}
		close(mypipe[0]);
		// usleep(1000);
	}
	else
	{
		rtc = 1;
	}

	if (rtc)
	{
		fprintf(stderr, "ALERT:" THE_FILTER "[%d]: cannot %s: %s\n",
			my_getpid(), rtc == 1? "pipe": "fdopen", strerror(errno));
		rtc = 1;
	}
	return rtc;
}

static void
run_sig_function(fl_init_parm const*fn, fl_parm *fl, int sig)
{
	fl_callback handler;
	switch (sig)
	{
		case SIGHUP:
			handler = fn->on_sighup;
			break;

		case SIGUSR1:
			handler = fn->on_sigusr1;
			break;

		case SIGUSR2:
			handler = fn->on_sigusr2;
			break;

		default:
			handler = NULL;
			break;
	}

	if (handler)
		(*handler)(fl);
}

static int fl_run_batchtest(fl_init_parm const*fn, fl_parm *fl)
{
	int rtc = 0;
	int mypipe[2];
	FILE *fp = NULL;
	int pending = 0, total = 0;
	unsigned sleep_arg = 0;

	fl->out = 1; // stdout

	while (!feof(stdin) && fl_keep_running())
	{
		char cmdbuf[1024];
		char *s, *es = NULL;

		if (sleep_arg == 0)
		{
			if (isatty(fileno(stdout)))
			{
				fprintf(stdout, "%d: ", total);
				fflush(stdout);
			}
			errno = 0;
			s = fgets(cmdbuf, sizeof cmdbuf, stdin);
			if (signal_hangup)
			{
				int const sig = signal_hangup;
				signal_hangup = 0;
				run_sig_function(fn, fl, sig);
			}
		}
		else
			s = NULL;
		if (s || sleep_arg)
		{
			int is_exit = 0;
			if (s)
			{
				es = s + strlen(s);
				while (--es >= s && isspace(*(unsigned char*)es))
					*es = 0;
				++es;
				is_exit = strcmp(s, "exit") == 0;
			}

			if (s == es || is_exit || sleep_arg)
			{
				if (pending)
				{
					fputc('\n', fp);
					fflush(fp);
					fl_runchild(fl);
					fclose(fp);
					close(mypipe[0]);
					close(mypipe[1]);
					pending = 0;
					++total;
					if (fl->verbose >= 8)
						fprintf(stdout, "%d child(ren) total, %d alive\n",
							total, live_children);
				}
				if (is_exit)
					break;
				if (sleep_arg)
				{
					sleep(sleep_arg);
					sleep_arg = 0;
					if (signal_hangup)
					{
						int const sig = signal_hangup;
						signal_hangup = 0;
						run_sig_function(fn, fl, sig);
					}
				}
			}
			else if (strcmp(s, "sigusr1") == 0)
				run_sig_function(fn, fl, SIGUSR1);
			else if (strcmp(s, "sigusr2") == 0)
				run_sig_function(fn, fl, SIGUSR2);
			else if (strcmp(s, "sighup") == 0)
				run_sig_function(fn, fl, SIGHUP);
			else if (strncmp(s, "test", 4) == 0)
			{
				fl_callback handler;
				switch (s[4])
				{
					case '1': handler = fn->test_fn1; break;
					case '2': handler = fn->test_fn2; break;
					case '3': handler = fn->test_fn3; break;
					case '4': handler = fn->test_fn4; break;
					default:
						handler = NULL;
						break;
				}
				if (handler)
					(*handler)(fl);
			}
			else if (strncmp(s, "sleep ", 6) == 0)
			{
				char *t = NULL;
				unsigned l = strtoul(s + 6, &t, 0);
				if (l == 0 || t == NULL || *t != 0)
					fputs("sleep requires a number\n", stderr);
				else
					sleep_arg = l;
			}
			else if (strcmp(s, "?") == 0) fputs(
			"filterlib batch test: recognized commands are:\n"
			" sigusr1, sigusr2, sighup   - simulate the corresponding signal\n"
			" test1, test2, test3, test4 - call app function, if defined\n"
			"                              (usually test1 prints configuration)\n"
			" sleep nn                   - sleep nn seconds or until signal\n"
			" exit                       - terminate batch testing\n"
			"unrecognized lines are interpreted as mail files and passed to the\n"
			"filter: mail file first, any number of ctl files until an empty line\n"
			"or one with a recognized command\n", stdout);

			else
			{
				if (fl->verbose >= 8)
					fprintf(stdout,
						"interpreted as %s file\n",
							pending == 0 ? "mail" : "ctl");
				if (pending == 0)
				{
					if ((rtc = pipe(mypipe)) == 0 &&
						(fp = fdopen(mypipe[1], "w")) != NULL)
							fl->in = mypipe[0];
					else
					{
						perror(rtc? "cannot pipe": "cannot fdopen");
						rtc = 1;
						break;
					}
				}

				fprintf(fp, "%s\n", s);
				++pending;
			}
		}
		else if (ferror(stdin))
		{
			int const handled = errno == EINTR || errno == EAGAIN;
			if (fl->verbose >= 8 || !handled)
				fprintf(stderr, "error reading stdin: %s\n",
					strerror(errno));
			if (!handled)
			{
				rtc = 1;
				break;
			}
			if (!feof(stdin))
				clearerr(stdin);
		}
	}

	if (pending)
	{
		fclose(fp);
		close(mypipe[0]);
		close(mypipe[1]);
	}

	if (fl->verbose >= 6)
		fprintf(stdout, "run %d total, %d still alive\n",
			total, live_children);

	return rtc;
}

static int fl_init_socket(fl_parm *fl, int allocal)
{
	assert(fl);
	assert(fl->argv0);

	int listensock = -1;

	char const *name = strrchr(fl->argv0, '/');
	if (name) ++name;
	else name = fl->argv0;

	static const char dir[] = FILTERSOCKETDIR,
		alldir[] = ALLFILTERSOCKETDIR;
	static_assert(sizeof alldir >= sizeof dir);
	size_t len = strlen(name) + 2 + sizeof alldir;

	char *sockname = malloc(len),
		*tmpsockname = malloc(len),
		*othername = malloc(len);
	if (sockname && tmpsockname && othername)
	{
		if (allocal)
		{
			strcat(strcat(strcpy(sockname, "."), "/s_"), name);
			strcat(strcat(strcpy(tmpsockname, "."), "/.s_"), name);
			strcat(strcat(strcpy(othername, "other"), "/s_"), name);
		}
		else if (fl->all_mode)
		{
			strcat(strcat(strcpy(sockname, alldir), "/"), name);
			strcat(strcat(strcpy(tmpsockname, alldir), "/."), name);
			strcat(strcat(strcpy(othername, dir), "/"), name);
		}
		else
		{
			strcat(strcat(strcpy(sockname, dir), "/"), name);
			strcat(strcat(strcpy(tmpsockname, dir), "/."), name);
			strcat(strcat(strcpy(othername, alldir), "/"), name);
		}

		struct sockaddr_un ssun;
		ssun.sun_family=AF_UNIX;
		strcpy(ssun.sun_path, tmpsockname);
		unlink(ssun.sun_path);
		if ((listensock=socket(PF_UNIX, SOCK_STREAM, 0)) < 0 ||
			bind(listensock, (struct sockaddr *)&ssun, sizeof(ssun)) < 0 ||
			listen(listensock, SOMAXCONN) < 0 ||
			chmod(ssun.sun_path, 0660) ||
			rename (tmpsockname, sockname) ||
			fcntl(listensock, F_SETFL, O_NDELAY) < 0)
		{
			fl_report(LOG_ALERT, "fl_init_socket failed: %s", strerror(errno));
			if (listensock >= 0)
			{
				close(listensock);
				listensock = -1;
			}
		}
		else if (unlink(othername) && errno != ENOENT)
			fl_report(LOG_ERR, "unlink(%s) failed: %s",
				othername, strerror(errno));

		if (listensock >= 0 && fl->verbose >= 6)
			fl_report(LOG_INFO, "listening on %s", sockname);
	}
	// else malloc failure on init...

	free(sockname);
	free(tmpsockname);
	free(othername);
	return listensock;
}

static void lf_init_completed(int sockfd)
{
	if (sockfd != 3)	close(3);
}

static int is_courierfilter(int verbose)
{
	int rtc = 1;
	struct stat stat;
	if (fstat(3, &stat) != 0)
	{
		if (errno == EBADF)
		{
			rtc = 0;
		}
		else if (verbose >= 1)
			fprintf(stderr, THE_FILTER ": cannot fstat 3: %s\n",
				strerror(errno));
	}
	else
	{
#if !defined(NDEBUG)
		fprintf(stderr, THE_FILTER ": fd 3 has mode=%lx size=%ld fstype=%.*s\n",
			(unsigned long)stat.st_mode,
			(unsigned long)stat.st_size,
#if HAVE_ST_FSTYPE_STRING
			(int)sizeof stat.st_fstype,
			stat.st_fstype);
#else
			6,
			"undef.");
#endif // HAVE_ST_FSTYPE_STRING
#endif // !defined(NDEBUG)

		if (stat.st_size > 1) // available to read
			rtc = 0;
	}

	if (rtc == 0)
		fprintf(stderr, THE_FILTER ": bad fd3: invalid call\n");
	return rtc;
}

int fl_main(fl_init_parm const*fn, void *parm,
	int argc, char *argv[], int all_mode, int verbose)
{
	int rtc = 0, alllocal = 0;
	fl_parm fl;

	memset(&fl, 0, sizeof fl);
	fl.parm = parm;
	fl.verbose = sig_verbose = verbose;
	fl.filter_fn = fn ? fn->filter_fn : NULL;
	fl.argv0 = argv[0]? argv[0]: THE_FILTER;
	fl.all_mode = all_mode != 0;

	for (int i = 1; i < argc; ++i)
	{
		char const *const arg = argv[i];
		if (strcmp(arg, "-t") == 0)
		{
			fl.testing = 1;
			fl_init_signal(&fl);
			if (fl.verbose >= 2)
				fprintf(stderr,
					THE_FILTER ": running test on %d files\n",
					argc - i - 1);
			set_log_no_pid(1);
			rtc = fl_runtest(&fl, argc - i - 1, argv + i + 1);
			break;
		}

		if (strcmp(arg, "--batch-test") == 0)
		{
			assert(fn);

			fl.batch_test = fl.testing = 1;
			fl_init_signal(&fl);
			if (isatty(fileno(stdout)))
				fprintf(stdout,
					THE_FILTER ": batch test. Type `?' for help.\n");
			set_log_no_pid(1);
			rtc = fl_run_batchtest(fn, &fl);
			break;
		}

		if (strcmp(arg, "--all-local") == 0)
			alllocal = 1;

		if (strcmp(arg, "--help") == 0)
		{
			fputs(
			/*  12345678901234567890123456 */
				"  -t file...              scan rest of args as mail file(s)\n"
				"  --batch-test            enter batch test mode\n"
				"  --all-local             enter local test mode\n",
					stdout);
			return 1;
		}
	}

	int servicing_child_pipe = -1;

	if (fl.testing == 0 &&
		/* argc == 1 &&  -- allow --all-local */
		is_courierfilter(fl.verbose)) /* install filter */
	{
		int listensock = -1;

		if (fl.verbose >= 3)
			fl_report(LOG_INFO, "running version " PACKAGE_VERSION);
		setsid();
		/*
		int rtc = setpgrp();

		if (rtc)
			fprintf(stderr, THE_FILTER ": cannot set process group\n");
		*/

		fl_init_signal(&fl);
		listensock = fl_init_socket(&fl, alllocal);

		if (listensock < 0)
			return 1;

		if (fn->init_complete)
			(*fn->init_complete)(&fl);

		lf_init_completed(listensock);

		sigprocmask(SIG_BLOCK, &fl.blockmask, &fl.allowset);
		for (;;)   /* installed filter's main loop */
		{
			int const sig = signal_hangup;

			/*
			** while the sig function runs, spawn a child to continue
			** servicing requests on the socket until sig function returns.
			*/
			if (sig != 0)
			{
				signal_hangup = 0;
				if (servicing_child_pipe >= 0) // killed by parent
					break;

				int child_pipe[2];
				if (pipe(child_pipe) == 0)
				{
					pid_t pid = my_fork("reloading");
					if (pid > 0) // parent
					{
						sigprocmask(SIG_SETMASK, &fl.allowset, NULL);
						close(child_pipe[1]);
						run_sig_function(fn, &fl, sig);
						if (kill(pid, SIGHUP) == 0)
							read(child_pipe[0], &child_pipe[1], 1);
						close(child_pipe[0]);
						sigprocmask(SIG_BLOCK, &fl.blockmask, NULL);
						if (signal_hangup)
							continue;
					}
					else if (pid == 0) // child
					{
						close(child_pipe[0]);
						servicing_child_pipe = child_pipe[1];
						live_children = 0; // these are siblings, actually
						if (fl.verbose >= 2)
							fl_report(LOG_INFO,
								"spawned child while parent reloads");
					}
					else
					{
						close(child_pipe[0]);
						close(child_pipe[1]);
					}
				}
				else
					fl_report(LOG_CRIT,
						"ignore reload signal after pipe error: %s",
							strerror(errno));
			}

			int fd = my_lf_accept(listensock, &fl.allowset);
			if (fd <= 0)
			{
				if (fd < 0) /* select interrupted */
				{
					assert(errno == EAGAIN || errno == EINTR);
					continue;
				}

				/* fd == 0 for clean shutdown */
				break;
			}
			signal_timed_out = signal_break = 0;
			sigprocmask(SIG_SETMASK, &fl.allowset, NULL);

			fl.in = fl.out = fd;
			fl_runchild(&fl);
			close(fd);

			sigprocmask(SIG_BLOCK, &fl.blockmask, NULL);
		} // end of loop
		sigprocmask(SIG_SETMASK, &fl.allowset, NULL);
	}

	if (fl.verbose && live_children == 0)
		fprintf(stderr, THE_FILTER "[%d]: exiting\n", (int)getpid());

	int wait_child = 5 + live_children;

	if (servicing_child_pipe >= 0)
		close(servicing_child_pipe);

	if (getenv("DEBUG_FILTER") && live_children == 1 &&
		fl_get_test_mode(&fl) == fl_testing)
	{
		fprintf(stderr, THE_FILTER
			": leaving the child running for dbg\n");
		return rtc;
	}

	while (live_children > 0 && wait_child >= 0 && signal_break == 0)
	{
		int nsec =  live_children*3;

		if (fl.batch_test == 0 && fl.verbose)
			fprintf(stderr, THE_FILTER "[%d]: waiting for %d child(ren)\n",
				my_getpid(), live_children);

		if (sleep(nsec) != 0)
			continue;

		// kill all processes in the group --should check getpgrp() == getpid()?
		if (fl.testing == 0 && servicing_child_pipe == -1) // setsid() was issued
		{
			kill(0, SIGTERM);
			if (--wait_child < 0)
			{
				fprintf(stderr, "WARN:"
					THE_FILTER
					"[%d]: leaving %d naughty child(ren) running\n",
					my_getpid(), live_children);
			}
		}
	}
	if (servicing_child_pipe >= 0)
		exit(rtc);

	return rtc;
}
