/*
** avfilter.c - written in milano by vesely on 19 oct 2001
** ClamAV for courier-mta global filters (used to use Sophos av)
*/
/*
Copyright (C) 2001-2025 Alessandro Vesely

This file is part of avfilter

avfilter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

avfilter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License version 3
along with avfilter.  If not, see <http://www.gnu.org/licenses/>.
*/
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <limits.h>
#include <stdbool.h>

#include <sys/types.h>
#include <sys/time.h>
#include <time.h>

#include <sys/wait.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
//#include <sys/mman.h>

#include <inttypes.h>
#include <syslog.h> // severity

/* ClamAV */
#include <clamav.h>

#include "log.h"
#include "parm.h"
#include "filter_msg.h"
#include "filterlib.h"
#include "avfiledefs.h"
#include "cstring.h"
#include "vb_fgets.h"
#include "scan_api.h"
#include "avfilter_scan.h"

// last
#include <assert.h>

#if !defined(PATH_MAX)
#if defined(MAX_PATH)
#define PATH_MAX MAX_PATH
#else
#define PATH_MAX 255
#endif
#endif

/******************************************************************************
utilities
******************************************************************************/

static const char *month_abbr(unsigned ndx)
{
	static const char *abbr[] =
	{"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	if (ndx >= sizeof abbr / sizeof abbr[0]) return "---";
	return abbr[ndx];
}

static char const*filter_msg_action_name(filter_msg_action action)
{
	static char const *const action_name[] =
		/* 0       1       2         3       4    */
		{ "none", "pass", "reject", "drop", "INVALID ACTION"};
	return action_name[(unsigned)action < 4U ? action : 4U];
}

char *hdrval(const char *a, const char *b)
// b must be without trailing ':'
// return pointer after column if headers match, NULL otherwise
{
	assert(a && b && strchr(b, ':') == NULL);
	
	int c, d;
	do c = *(unsigned char const*)a++, d = *(unsigned char const*)b++;
	while (c != 0 && d != 0 && (c == d || tolower(c) == tolower(d)));
	
	if (d != 0 || c == 0)
		return NULL;

	while (c != ':')
		if (!isspace(c) || (c = *(unsigned char const*)a++) == 0)
			return NULL;

	return (char*)a;
}

static int wildstr(const char *card, const char *name)
// 0 == match, 1 == no-match
{
	assert(card != 0 && name != 0);

	unsigned char c = *card, n = *name;
	while(c != 0 && n != 0)
		if (c == n || c == '?')
		{
			c = *++card;
			n = *++name;
		}
		else if (c == '*')
		{
			c = card[1];
			while (n != c)
				if ((n = *++name) == 0) return c == 0 ? 0 : 1;
			// matching char found? if rest doesn't match then skip it
			if (wildstr(card + 1, name) == 0) return 0;
			c = '*';
			n = *++name;
		}
		else return 1;
	return c == n || (c == '*' && card[1] == 0) ? 0 : 1;
}

static int wild_in_charray(charray *a, char const *name)
// 0 == not-found, 1 == found
{
	if (a && a->count && name)
		for (size_t i = 0; i < a->count; ++i)
			if (wildstr(a->array[i], name) == 0)
				return 1;

	return 0;
}

typedef enum dir_check
{
	dont_create_dir = 0,
	create_dir = 1,
	check_read_access = 2,
	check_write_access = 4,
	check_owner = 8
} dir_check;

static inline int my_access(const char *pathname, int mode)
{
#if HAVE_EUIDACCESS
	return euidaccess(pathname, mode);
#else
	return access(pathname, mode);
#endif
}

static int check_directory(const char *dir_name,
	const char *dir, dir_check tmpcreate, FILE *report, int verbose)
{
	assert(report);

	if (dir == NULL)
		return 0;

	bool to_be_created = strncmp(dir, "/tmp/", 5) == 0;
	struct stat s;

	if ((tmpcreate & create_dir) != 0 &&
		to_be_created &&
		stat(dir, &s) != 0 && errno == ENOENT)
	{
		if (mkdir(dir, 0770) == 0)
			filelog(report, LOG_INFO, "%s directory %s created",
				dir_name, dir);
		else if (verbose >= 1)
			filelog(report, LOG_ERR, "mkdir %s failed: %s",
				dir, strerror(errno));
	}

	errno = 0;
	int rtc = stat(dir, &s);
	if (rtc != 0 || !S_ISDIR(s.st_mode))
	{
		/*
		* on testing, don't create directories
		* so cannot check them
		*/
		if (rtc && errno == ENOENT && (tmpcreate & create_dir) == 0 && to_be_created)
			rtc = 0;
		else
		{
			rtc = 1;
			if (verbose >= 1)
				filelog(report, LOG_ERR,
					"unsuitable directory %s: %s",
					dir,
					errno ? strerror(errno): "not a directory");
		}
	}
	else
	{
		if ((tmpcreate & check_read_access) != 0)
		{
			if (my_access(dir, R_OK|X_OK) != 0)
				if (verbose >= 1)
					filelog(report, LOG_ERR, "unreadable %s directory %s: %s",
						dir_name, dir, strerror(errno));
		}

		if ((tmpcreate & check_write_access) != 0)
		{
			if (my_access(dir, R_OK|W_OK|X_OK) != 0)
				if (verbose >= 1)
					filelog(report, LOG_ERR, "unwritable %s directory %s: %s",
						dir_name, dir, strerror(errno));
		}

		if ((tmpcreate & check_owner) != 0)
		{
			if ((S_IRWXU & s.st_mode) != S_IRWXU /* owner has read, write, and execute permission */ ||
				(S_IRWXO & s.st_mode) != 0 /* other has read, write, and execute permission */ ||
				s.st_uid != MAILUID) /*ownership */
			{
				int sev = (rtc || !S_ISDIR(s.st_mode)) ? LOG_ERR : LOG_WARNING;
#if HAVE_EUIDACCESS
				if (sev != LOG_ERR && euidaccess(dir, R_OK|W_OK|X_OK) != 0)
					sev = LOG_ERR;
#endif
				rtc = sev == LOG_ERR;

				if (verbose >= 1)
					filelog(report, sev,
						"unsuitable %s directory %s: %s",
						dir_name, dir,
							s.st_uid != MAILUID? "not owned by " MAILUSER:
							(S_IRWXU & s.st_mode) != S_IRWXU ? "not rwx by " MAILUSER:
							(S_IRWXO & s.st_mode) != 0? "world accessible" : "uh?");
			}
		}
	}

	return rtc;
}

static int
check_consistency(parm_t *a, dir_check tmp, FILE *logfile, int verbose)
{
	assert(a);

	int err = 0;

	if (!a->applied_parm_defaults)
		apply_parm_defaults(a);

	if (a->pua && a->pua->count > 0 &&
		(a->load_options & CL_DB_PUA_EXCLUDE) == 0 &&
		(a->load_options & CL_DB_PUA_INCLUDE) == 0)
	{
		if (verbose >= 1)
			filelog(logfile, LOG_WARNING,
				"pua array is defined, but neither exclude_ nor include_pua is");
	}

	if ((a->pua == NULL || a->pua->count == 0) &&
		((a->load_options & CL_DB_PUA_EXCLUDE) != 0 ||
		((a->load_options & CL_DB_PUA_INCLUDE) != 0)))
	{
		if (verbose >= 1)
			filelog(logfile, LOG_ERR,
				"%sclude_pua is defined, but pua array is not",
					(a->load_options & CL_DB_PUA_EXCLUDE) != 0? "ex": "in");
		err = 1;
	}

	if ((a->load_options & CL_DB_PUA_EXCLUDE) != 0 &&
		(a->load_options & CL_DB_PUA_INCLUDE) != 0)
	{
		if (verbose >= 1)
			filelog(logfile, LOG_ERR,
				"exclude_pua and include_pua cannot be defined simultaneously");
		err = 1;
	}

	err |= check_directory("save_virus",
		a->save_virus, err? dont_create_dir : tmp | check_owner, logfile, verbose);
	err |= check_directory("save_missed",
		a->save_missed, err? dont_create_dir: tmp | check_owner, logfile, verbose);
	err |= check_directory("tempdir",
		a->tempdir, err? dont_create_dir: tmp | check_write_access, logfile, verbose);

	check_directory("pid dir",
		a->piddir? a->piddir: AVFILTER_PID_DIR, check_write_access, logfile, verbose);

	check_directory("database",
		a->database? a->database: cl_retdbdir(), check_read_access, logfile, verbose);

	char *s = a->virus_header;
	if (s)
	{
		size_t len = 0, bad = 0;
		int ch;
		while ((ch = *(unsigned char*)s++) != 0)
		{
			len += 1;
			bad += ch < 33 || ch > 126 || ch == ':';
		}
		if (len <= 0 || len > 50)
		{
			if (verbose >= 1)
				filelog(logfile, LOG_ERR,
					"virus_header is %s", len? "too long (max 50)": "empty");
			err = 1;
		}
		if (bad)
		{
			if (verbose >= 1)
				filelog(logfile, LOG_ERR,
					"virus_header has %zu non-printable, non-ASCII, or ':' char(s)",
						bad);
			err = 1;
		}
	}

	if ((a->scan_options.parse & CL_SCAN_PARSE_MAIL) == 0) // no nonsense
	{
		err = 1;
		filelog(logfile, LOG_ERR, "scan_mail disabled?!");
	}

	return err;
}

/******************************************************************************
specific
******************************************************************************/

typedef struct virus_load_stats
{
	time_t wall_date, mono_date;
	unsigned count;
} virus_load_stats;

typedef struct avfilter_parm_struct
{
	char *config_file;
	FILE *report; // reload pipe, or virus save

	/* other values retained on reloading */
	struct cl_engine *clamav;
	virus_load_stats virus_load;

	parm_t a; // values read from config_file
	uint32_t x55aa5a5a;
	char pid_created;
	char err; /* error parsing config file */
} av_parm;

static av_parm* get_parm(fl_parm *fl)
{
	av_parm* parm = (av_parm*) fl_get_parm(fl);
	if (parm == NULL || parm->x55aa5a5a != 0x55aa5a5a)
		abort();
	return parm;
}

/*
* cl_engine_free() may take an inordinate amount of time to complete,
* so, when the program is just about to exit,  do it only when compiled
* with -fsanitize-address, which would detect leaks.  Exiting the
* program automatically frees everything quickly.
*/
#if defined(__has_feature)
#if __has_feature(address_sanitizer)
#define DO_CL_ENGINE_FREE
#endif
#else
#if __SANITIZE_ADDRESS__
#define DO_CL_ENGINE_FREE
#endif
#endif

static void my_cl_engine_free(int verbose,
	struct cl_engine *clamav, char const *whence)
{
	struct timespec before, after;
	int rtc = clock_gettime(CLOCK_MONOTONIC, &before);
	cl_engine_free(clamav);
	rtc |= clock_gettime(CLOCK_MONOTONIC, &after);
	if (rtc == 0 && verbose >= 2)
	{
		struct timeval b, a, diff;
		TIMESPEC_TO_TIMEVAL(&a, &after);
		TIMESPEC_TO_TIMEVAL(&b, &before);
		timersub(&a, &b, &diff);
		filelog(stderr, LOG_INFO,
			"cl_engine_free() in %ld.%06lds at %s",
			diff.tv_sec, diff.tv_usec, whence);
	}
}

static int clamav_load_n_report(av_parm *parm)
// return 0 if ok, 1 on load error
{
	assert(parm);

	unsigned err = 0;
	struct timespec before, after;
	int time_rtc = clock_gettime(CLOCK_MONOTONIC, &before);

	FILE *logfile = set_log_file(NULL);   //
	char const *clamav_ver = cl_retver(); //  E.g. 0.100.0-beta
	int clamav_ver_n = 0;
	int ch, ch1;
	char const *s = clamav_ver;
	while ((ch1 = *(unsigned char*)s++) != 0 && !isdigit(ch1))
		continue;
	if ((ch1 == '0' || ch1 == '1') && (ch = *(unsigned char*)s++) == '.')
	{
		while ((ch = *(unsigned char*)s++) != 0 && isdigit(ch))
		{
			clamav_ver_n += ch - '0';
			clamav_ver_n *= 10;
		}
		if (ch1 == '1')
			clamav_ver_n += 10000;
	}
	if (clamav_ver_n < 1010)
	{
		filelog(logfile, LOG_ALERT,
			"Bad libclamav version %s, need 101 or better", clamav_ver);
		return 1;
	}

	/*
	* Load ClamAV (new engine, config, load, compile)
	*/
	struct cl_engine *clamav = cl_engine_new();
	if (clamav == NULL)
	{
		filelog(logfile, LOG_ALERT, "cannot create ClamAV engine");
		return 1;
	}

	int rtc = set_engine_values(&parm->a, clamav);
	if (rtc)
		err = 1;

	cl_engine_set_clcb_virus_found(clamav, virus_found_cb);

	unsigned int sigs = 0;
	const char *dbdir = parm->a.database? parm->a.database: cl_retdbdir();

	if ((rtc = cl_load(dbdir, clamav, &sigs, parm->a.load_options)) != CL_SUCCESS)
	{
		filelog(logfile, LOG_ERR, "cl_load error: %s", cl_strerror(rtc));
		err |= 2;
	}

	if ((rtc = cl_engine_compile(clamav)) != CL_SUCCESS)
	{
		filelog(logfile, LOG_ERR, "cl_compile error: %s", cl_strerror(rtc));
		err |= 4;
	}

	if (err)
	{
		filelog(logfile, LOG_ALERT,
			"ClamAV not %sloaded: set_engine %s, cl_load %s, cl_compile %s",
			parm->clamav? "re": "",
			err&1? "bad": "ok",
			err&2? "bad": "ok",
			err&4? "bad": "ok");
		my_cl_engine_free(0, clamav, "err1");
		return 1;
	}

	time_rtc |= clock_gettime(CLOCK_MONOTONIC, &after);

	if (parm->clamav)
		my_cl_engine_free(parm->a.verbose, parm->clamav, "reloaded");
	parm->clamav = clamav;

	/*
	* Ask for the engine information - virus engine and data version number,
	* and the number of viruses.
	*/
	if (parm->a.verbose >= 2)
	{
		struct tm tm = {0}; // compiler happy
		if (time_rtc == 0)
		{
			struct timeval b, a, diff;
			TIMESPEC_TO_TIMEVAL(&a, &after);
			TIMESPEC_TO_TIMEVAL(&b, &before);
			timersub(&a, &b, &diff);
			filelog(logfile, LOG_INFO,
				"virus data loaded in: %ld.%06lds",
				diff.tv_sec, diff.tv_usec);
		}

		if (parm->virus_load.wall_date && parm->virus_load.mono_date)
		{
			tm = *localtime(&parm->virus_load.wall_date);
			unsigned elapsed = before.tv_sec - parm->virus_load.mono_date;
			unsigned dd, hh, mm;
			char buf[32], *p = &buf[0];

			dd = elapsed / (24UL * 3600UL);
			if (dd)
			{
				elapsed -= dd * 24UL * 3600UL;
				p += sprintf(p, "%u day(s) ", dd);
			}
			hh = elapsed / 3600UL;
			if (hh || dd)
			{
				elapsed -= hh * 3600UL;
				p += sprintf(p, "%uh ", hh);
			}
			mm = elapsed / 60UL;
			elapsed -= mm * 60UL;
			sprintf(p, "%um ", mm);

			filelog(logfile, LOG_INFO,
				"previous load was:    %d %s %d (%s%us ago)",
				tm.tm_mday, month_abbr(tm.tm_mon), tm.tm_year + 1900,
				buf, elapsed);
		}
		filelog(logfile, LOG_INFO, "Virus engine version: %s (f=%u)",
			cl_retver(), cl_retflevel());

		int dataerr = 0;
		time_t db_time = (time_t)
			cl_engine_get_num(clamav, CL_ENGINE_DB_TIME, &dataerr);
		if (dataerr == 0) tm = *localtime(&db_time);

		uint32_t const db_version = (uint32_t)
			cl_engine_get_num(clamav, CL_ENGINE_DB_VERSION, &dataerr);

		if (dataerr == 0)
		{
			filelog(logfile, LOG_INFO, "Virus data version:   %"PRIu32,
				db_version);
			filelog(logfile, LOG_INFO, "Virus data date:      %d %s %d",
				tm.tm_mday, month_abbr(tm.tm_mon), tm.tm_year + 1900);
		}
	}
	if (parm->a.verbose >= 1)
		filelog(logfile, LOG_INFO, "No. of viruses:       %u", sigs);
	if (parm->a.verbose >= 2 && parm->virus_load.count)
		filelog(logfile, LOG_INFO, "previous count was:   %u (%d increment)",
			parm->virus_load.count, (int)(sigs - parm->virus_load.count));
	parm->virus_load.count = sigs;
	parm->virus_load.mono_date = before.tv_sec;
	time(&parm->virus_load.wall_date);

	return 0;
}

static int save_callback(void *v, char const *r)
{
	av_parm *parm = (av_parm*)v;

	assert(parm);
	assert(parm->report);

	fprintf(parm->report, "%s\n", r);
	return 0;
}

static int
save_file(fl_parm *fl, char const *name, int virus)
// 0 = no error, otherwise logged
{
	assert(fl);

	static const char templ[] = "/avf-";

	av_parm *parm = get_parm(fl);
	char const *dir = virus? parm->a.save_virus: parm->a.save_missed;

	assert(parm->report == NULL);
	assert(dir);

	unsigned const dir_len = strlen(dir);
	unsigned const name_len = strlen(name);
	unsigned const namesize = dir_len + sizeof templ + name_len + 10;
	char *const fname = (char*)malloc(namesize);
	char const *error;
	int fno /* compiler happy */ = 0;
	int rtc = 1;

	if (fname == NULL)
		error = "malloc fail";
	else
	{
		memcpy(fname, dir, dir_len);
		memcpy(fname + dir_len, templ, sizeof templ);

		char *p = fname + dir_len + sizeof templ - 1,
		// max space available for name; 6 for XXXXXX
			*const end = fname + PATH_MAX - 6;

		if (p > end)
			error = "path too long";
		else
		{
			char const *s = name;
			while (p < end)
			{
				int ch = *(unsigned char*)s++;
				if (ch == 0)
					break;

				if (!isalnum(ch) && ch != '.' && ch != '-')
					ch = '_';
				*p++ = ch;
			}
			if (p < end)
				*p++ = '-';
			//         123456
			strcpy(p, "XXXXXX");
			assert(strlen(fname) == dir_len + sizeof templ - 1
				+ (name_len < (PATH_MAX - (dir_len + sizeof templ - 1) - 6) ?
					name_len : (PATH_MAX - (dir_len + sizeof templ - 1) - 6))
				+ (name_len < (PATH_MAX - (dir_len + sizeof templ - 1) - 6) ? 1 : 0)
				+ 6);
			assert(strlen(fname) < namesize);

			fno = mkstemp(fname);
			if (fno == -1)
				error = "mkstemp fails";
			else
				error = NULL;
		}
	}

	char *sender = NULL;
	if (virus && (sender = fl_get_sender(fl)) == NULL)
		error = "cannot get sender";

	if (error)
	{
		filelog(stderr, LOG_ERR, "Unable to save %.*s: %s",
			PATH_MAX, fname ? fname: "(null)", error);
	}
	else
	{
		FILE *fps = fdopen(fno, "w+");
		char buf[4096];
		size_t r = virus;
		if (fps)
		{
			/* for viruses, write pre-header */
			if (virus)
			{
				fprintf(fps, "%s\n", sender);
				parm->report = fps;
				if (rcpt_iterate(fl, save_callback) == 0)
					r = 0;
				fputc('\n', fps); // empty line ends pre-header
				parm->report = NULL;
			}

			/* copy mail file */
			FILE *fp = fl_get_file(fl);
			fseek(fp, 0, SEEK_SET);
			if (r == 0)
			{
				while ((r = fread(buf, 1, sizeof buf, fp)) > 0)
					if (fwrite(buf, r, 1, fps) != 1)
						break;
				r = (ferror(fp) || ferror(fps)) ? 1 : 0;
			}
			fclose(fps);
		}
		else
		{
			close(fno);
			r = 1;
		}
		if (r)
		{
			filelog(stderr, LOG_ERR, "Unable to save %s: %s",
				fname, strerror(errno));
			unlink(fname);
		}
		else
		{
			rtc = 0;
			if (parm->a.verbose >= 2)
				filelog(stderr, LOG_INFO, "Saved %s file in %s",
					virus ? "virus" : "missed", fname);
		}
	}
	free(sender);
	free(fname);
	return rtc;
}

static filter_msg_action virus_action(av_parm *parm, char const *name)
{
	/*
	* the reverse order is equivalent to assigning the most "severe"
	* action after searching all the arrays.
	*/
	  
	filter_msg_action found = msg_none;
	if (wild_in_charray(parm->a.drop, name))
		found = msg_drop;
	else if (wild_in_charray(parm->a.reject, name))
		found = msg_reject;
	else if (wild_in_charray(parm->a.pass, name))
		found = msg_pass;

	if (parm->a.verbose >= 2)
		filelog(stderr, LOG_INFO,
			"found virus \"%s\"; specific action: %s",
				name, filter_msg_action_name(found));

	return found;
}

static int check_config_action(av_parm *parm)
{
	assert(parm);

	uint8_t verbose = parm->a.verbose;
	parm->a.verbose = 0; // avoid "found virus" messages

	char buf[1024], *s;
	while ((s = fgets(buf, sizeof buf, stdin)) != NULL)
	{
		int ch;
		while ((ch = *(unsigned char*)s) != 0 && isspace(ch))
			++s;
		char *e = s + strlen(s) - 1;
		while (e >= s && isspace(ch = *(unsigned char*)e))
			*e-- = 0;

		filter_msg_action a;
		if (s <= e && (a = virus_action(parm, s)) > msg_none)
			printf("%s -> %s\n", s, filter_msg_action_name(a));
	}

	parm->a.verbose = verbose;
	return 0;
}

#define ONE_WASNT_FOUND 0xdead
static int recipient_match_cb(void *v, char const *r)
{
	av_parm *parm = (av_parm*)v;
	if (wild_in_charray(parm->a.pass_recipient, r) == 0) // not found
		return ONE_WASNT_FOUND;
	return 0;
}

static filter_msg_action
recipient_match(fl_parm *fl, filter_msg_action action)
{
	int rtc = rcpt_iterate(fl, recipient_match_cb);
	filter_msg_action next;

	switch (rtc)
	{
		case 0: // all recipients matched
			next = msg_pass;
			break;

		case ONE_WASNT_FOUND: // no override
			next = action;
			break;

		default: // errors occurred
			next = msg_none;
			break;
	}

	av_parm *parm = get_parm(fl);
	if (parm->a.verbose >= 2)
		filelog(stderr, LOG_INFO, "action %s to %s after recipient check",
			filter_msg_action_name(action),
			filter_msg_action_name(next));

	return next;
}

static cstring *append_he(cstring *he, char const *more)
{
	if (he && more)
	{
		char *fold = strrchr(he->data, '\n');
		if (fold)
			++fold;
		else
			fold = he->data;

		size_t len = strlen(fold),
			morelen = strlen(more);

		if (len + morelen + 1 > 78)
			he = cstr_addblob(he, "\n  ", 3);
		else
			he = cstr_addch(he, ' ');

		if (he) he = cstr_addblob(he, more, morelen);
	}
	return he;
}

static int rewrite(fl_parm *fl, char const *add_header)
/**
* Rewrite the file, Old-ing out any occurrence of a.virus_header
* if add_header, add it as the content of a true a.virus_header
* return 0 if successful
*/
{
	assert(fl);

	av_parm *parm = get_parm(fl);

	assert(parm);
	assert(parm->a.virus_header);

	FILE *fp = fl_get_file(fl);
	if (fp == NULL)
	{
		filelog(stderr, LOG_ERR, "data fp is NULL");
		return 1;
	}

	if (fseek(fp, 0, SEEK_SET))
	{
		filelog(stderr, LOG_ERR, "bad seek on data fp: %s", strerror(errno));
		return 1;
	}

	var_buf vb;
	if (vb_init(&vb))
	{
		filelog(stderr, LOG_ALERT, "MEMORY FAULT");
		return 1;
	}

	FILE *fp_out = NULL;
	if (add_header)
	{
		fp_out = fl_get_write_file(fl);
		if (fp_out == NULL)
		{
			vb_clean(&vb);
			return 1;
		}

		fputs(add_header, fp_out);
	}

	int rtc = 0;
	size_t keep = 0;
	char const * const my_header = parm->a.virus_header;

	for (;;)
	{
		char *p = vb_fgets(&vb, keep, fp);
		char *eol = p? strchr(p, '\n'): NULL;

		if (eol == NULL)
		{
			filelog(stderr, LOG_CRIT,
				"header too long (%.20s...)", vb_what(&vb, fp));
			rtc = -1;
			break;
		}

		int const next = eol >= p? fgetc(fp): '\n';
		int const cont = next != EOF && next != '\n';
		char *const start = vb.buf;
		if (cont && isspace(next)) // wrapped
		{
			*++eol = next;
			keep = eol + 1 - start;
			continue;
		}

		/*
		* full 0-terminated header field, including trailing \n, is in buffer;
		* if it is my_header, we need to rename it.
		*/
		if (hdrval(start, my_header))
		{
			if (fp_out == NULL)
			{
				if (fseek(fp, 0, SEEK_SET))
				{
					filelog(stderr, LOG_ERR, "cannot seek on data fp: %s",
						strerror(errno));
					rtc = -1;
					break;
				}

				if ((fp_out = fl_get_write_file(fl)))
				{
					rtc = -1;
					break;
				}

				keep = 0;
				continue;
			}

			fputs("Old-", fp_out); // rename existing field
		}

		/*
		* copy the field as is
		*/
		if (fp_out && fwrite(start, eol + 1 - start, 1, fp_out) != 1)
		{
			rtc = -1;
			break;
		}


		/*
		* continue header processing?
		*/
		if (!cont)
		{
			if (fp_out && next != EOF && fputc(next, fp_out) != next)
				rtc = -1;
			break;
		}

		start[0] = next;
		keep = 1;
	}

	/*
	* end of header; copy the rest if anything was written
	*/
	if (fp_out)
	{
		if (rtc == 0)
		{
			size_t bufsize = vb.alloc > 8192? 8192: vb.alloc, r;

			while ((r = fread(vb.buf, 1, bufsize, fp)) > 0)
				if (fwrite(vb.buf, r, 1, fp_out) != 1)
					break;

			if (r > 0 || ferror(fp))
				rtc = -1;
		}

		if (rtc == 0)
			rtc = fl_commit_write(fl);
		else
		{
			if (ferror(fp) || ferror(fp_out))
				filelog(stderr, LOG_CRIT,
					"file I/O error%s%s: %s",
					ferror(fp)? " read": "",
					ferror(fp_out)? " write": "",
					strerror(errno));
			fl_rollback_write(fl);
		}
	}

	vb_clean(&vb);
	return rtc;
}

static int add_virus_header(fl_parm *fl, virus_name_list *vnl)
// 0 = success
{
	assert(fl);

	av_parm *parm = get_parm(fl);

	assert(parm);
	assert(parm->a.virus_header);
	assert(strlen(parm->a.virus_header) <= 78); // Section 2.1.1 of RFC 5322

	cstring *he = cstr_init(80);
	if (he)
		he = cstr_addstr(he, parm->a.virus_header);
	if (he)
		he = cstr_addch(he, ':');

	for (virus_name_list *l = vnl; l != NULL; l = l->next)
		he = append_he(he, l->name);

	if (he)
		he = cstr_addch(he, '\n');

	if (he == NULL)
		return 1;

	int rtc = rewrite(fl, he->data);
	free(he);
	return rtc;
}

static cstring *
compose_virus_name_list(char const *fname,
	virus_name_list *vnl, char const *rtc)
{
	if (vnl && vnl->name[0])
	{
		cstring *s = cstr_from_string(rtc);
		if (s) s = cstr_addstr(s, fname);
		if (s) s = cstr_addstr(s, ":\n");
		while (vnl)
		{
			if (s) s = cstr_addch(s, '\t');
			if (s) s = cstr_addstr(s, vnl->name);
			if (s) s = cstr_addch(s, '\n');
			vnl = vnl->next;
		}
		if (s == NULL)
			filelog(stderr, LOG_CRIT, "memory fault");
		return s;
	}
	return NULL;
}

static void avfilter(fl_parm *fl) // main child function
{
	static char const*const resp_ok = "250 Ok.\n";
	static char const*const resp_drop = "050 Viral message dropped.\n";
	static char const*const resp_bad = "451 system malfunction in avfilter\n";
	// static char const*const resp_broken_archive = "551 broken zip attached\n";
	static char const*const resp_virus = "551 virus found\n";
	static char const*const hack_resp_clean = "0\n";
	static char const*const hack_resp_virus = "1\n";

	av_parm *parm = get_parm(fl);
	FILE *fp = fl_get_file(fl);

	char const *rtc = NULL;
	cstring *hack_rtc = NULL;
	int missed = 0;

	// seed is needed for forked programs.
	// /proc/sys/kernel/pid_max is usually 32768, so we could << 16...
	srand(getpid() << 1 + time(0));

	virus_name_list *vnl = NULL;
	char const *const fname = fl_get_filename(fl);

	int cl = my_scandesc(fileno(fp), fname, &vnl, parm->clamav, &parm->a.scan_options);
	/*
	* Hack to scan files also out of Courier.
	* the authsender (^i.*) is the unique code uniq_hack plus params
	* response is: 1st line return code (0=ok, 1=virus, errors...)
	*/
#define NOT_A_CLAMAV_CODE CL_ELAST_ERROR +10000

	char *authsender = fl_get_authsender(fl);
	if (authsender && strncmp(authsender, hack_uniq, sizeof hack_uniq - 1) == 0)
	{
		char *params = authsender + sizeof hack_uniq - 1;

		int verbose = 0;
		if (strncmp(params, hack_verbose, sizeof hack_verbose - 1) == 0)
		{
			verbose = 1;
			// params += sizeof hack_verbose - 1;
		}

		if (cl == CL_CLEAN)
		{
			cl = NOT_A_CLAMAV_CODE; // skip the switch below
			rtc = hack_resp_clean;
			if (parm->a.verbose >= 8)
				filelog(stderr, LOG_INFO, "hack_check: %s: OK", fname);
		}
		else if (cl == CL_VIRUS)
		{
			cl = NOT_A_CLAMAV_CODE;
			rtc = hack_resp_virus;
			if (verbose)
				hack_rtc = compose_virus_name_list(fname, vnl, rtc);
			if (parm->a.verbose >= 8)
				filelog(stderr, LOG_INFO, "hack_check: %s: BAD", fname);
		}
		else if (parm->a.verbose >= 1)
		{
			// CL_ error codes get SMTP response, etc..
			filelog(stderr, LOG_ERR, "hack_check: %s: CLAMAV ERROR", fname);
		}
	}

	switch (cl)
	{
		case CL_CLEAN:
			rtc = resp_ok;
			if (parm->a.virus_header && rewrite(fl, NULL))
			{
				rtc = resp_bad;
				if (parm->a.verbose >= 1)
					filelog(stderr, LOG_ERR,
						"deferred clean file; cannot check header");
			}
			else if (parm->a.verbose >= 8)
				filelog(stderr, LOG_INFO,
					"found clean file; action: pass");
			break;

		case CL_VIRUS:
		{
			filter_msg_action action = msg_none;
			char const *name = NULL;
			for (virus_name_list *l = vnl; l != NULL; l = l->next)
			{
				filter_msg_action a = virus_action(parm, l->name);
				if (a > action)
				{
					action = a;
					name = l->name;
				}
			}

			/*
			** if no specific action was configured, apply default action
			*/
			if (action <= msg_none)
			{
				name = vnl->name; // last virus found
				action = parm->a.action_default;
				if (action <= msg_none)
					action = msg_reject;

				if (parm->a.verbose >= 2 && action > msg_none)
					filelog(stderr, LOG_INFO, "default action: %s",
						filter_msg_action_name(action));
			}

			/*
			** pass_recipient overrides action, if any
			** (msg_none stands for failure from here on)
			*/
			if (parm->a.pass_recipient)
				action = recipient_match(fl, action);

			/*
			** always_pass overrides any other action
			*/
			if (parm->a.always_pass && action != msg_pass)
			{
				if (authsender)
				{
					if (wild_in_charray(parm->a.always_pass, authsender))
					{
						if (parm->a.verbose >= 2)
							filelog(stderr, LOG_INFO,
								"action %s to %s after authorized user %s",
								filter_msg_action_name(action),
								filter_msg_action_name(msg_pass),
								authsender);
						action = msg_pass;
					}
				}
			}


			int saved = 0;

			switch (action)
			{
				default:
				case msg_none:
					rtc = resp_bad;
					break;

				case msg_pass:
					rtc = resp_ok;
					if (name && parm->a.save_virus &&
						parm->a.save_only_if_drop == 0 &&
							save_file(fl, name, 1) == 0)
								saved = 1; // saved before rewrite
					if (parm->a.virus_header &&
						add_virus_header(fl, vnl))
							rtc = resp_bad;
					break;

				case msg_reject:
					if (name && parm->a.save_virus &&
						parm->a.save_only_if_drop == 0 &&
							save_file(fl, name, 1) == 0)
								saved = 1; // saved before reject
					rtc = resp_virus;
					break;

				case msg_drop:
					rtc = resp_drop;
					if (name && parm->a.save_virus &&
						save_file(fl, name, 1) == 0)
							saved = 1; // saved before drop (filterlib bug)
					if (fl_drop_message(fl))
						rtc = resp_bad;
					break;
			}

			if (parm->a.verbose >= 1)
				filelog(stderr, LOG_INFO,
					"found virus %s; final action %s; %ssaved",
						name,
						rtc == resp_bad? "deferred": filter_msg_action_name(action),
						saved? "": "not ");
			break;
		}

#if 0
		case CL_EUNPACK:
		/*
		This is never returned to caller, presumably to mean that scanning
		the overall container was completed successfully.
		*/
			rtc = parm->a.reject_broken_archive? resp_broken_archive: resp_ok;
			if (parm->a.verbose >= 1)
				filelog(stderr, LOG_INFO,
					"found broken archive; specific action: %s",
						rtc == resp_ok? "pass": "reject");
			break;
#endif

		/*
		TODO: check which ones are likely to be temporary errors
		      offer configurable alternatives
		*/
		case CL_EOPEN:
		case CL_ECREAT:
		case CL_EUNLINK:
		case CL_ESTAT:
		case CL_EREAD:
		case CL_ESEEK:
		case CL_EWRITE:
		case CL_EDUP:
		case CL_EACCES:
		case CL_ETMPFILE:
		case CL_ETMPDIR:
		case CL_EMAP:
		case CL_EMEM:
		case CL_ETIMEOUT:
			filelog(stderr, LOG_CRIT,
				"temporary clamav problem %s (%d)", cl_strerror(cl), cl);
			missed = 1;
			rtc = resp_bad;
			break;

		case NOT_A_CLAMAV_CODE: // hack done above...
			break;

		default:
			filelog(stderr, LOG_ALERT,
				"unexpected clamav error %s (%d)", cl_strerror(cl), cl);
			missed = 1;
			rtc = resp_ok;
			break;
	}

	/*
	** missed file
	*/
	if (missed && parm->a.save_missed)
	{
		char *name = strdup(cl_strerror(cl));
		if (name)
		{
			char *s = name;
			int ch;
			while ((ch = *(unsigned char*)s) != 0)
			{
				if (!isalnum(ch))
					*s = '.';
				++s;
			}
			save_file(fl, name, 0);
			free(name);
		}
	}

	free(authsender);
	fl_pass_message(fl, hack_rtc? cstr_get(hack_rtc, resp_bad): rtc);
	free(hack_rtc);
	free_virus_name_list(vnl);

#if defined DO_CL_ENGINE_FREE
	my_cl_engine_free(parm->a.verbose, parm->clamav, "child");  // happy LeakSanitizer
#endif
}

static void open_report_pipe(fl_parm* fl)
{
	assert(fl);
	char const *pipe_fname = fl_get_test_mode(fl) == fl_no_test?
		AVFILTER_PIPE_FILE: AVFILTER_PIPE_TEST;
	av_parm *parm = get_parm(fl);
	assert(parm);

	int const fd = open(pipe_fname, O_WRONLY
#if defined(O_NONBLOCK)
		| O_NONBLOCK
#elif defined(O_NDELAY)
		| O_NDELAY
#else
#error neither O_NONBLOCK nor O_NDELAY defined
#endif
			);
	if (fd < 0)
	{
		/* ENXIO if nobody reading the pipe, assume signal not by avfilter_sig
		** and file was stale */
		int severity = errno == ENXIO || errno == ENOENT ? LOG_INFO: LOG_ERR;
		filelog(stderr, severity, "cannot open %s: %s",
			pipe_fname, strerror(errno));
		parm->report = NULL;
	}
	else if (parm == NULL || (parm->report = fdopen(fd, "w")) == NULL)
	{
		filelog(stderr, LOG_ERR, "cannot fdopen %s on fd %d: %s",
			pipe_fname, fd, strerror(errno));
		close(fd);
	}
	else
		set_log_file(parm->report);
}

static void close_report_pipe(av_parm *parm)
{
	if (parm && parm->report)
	{
		fputs("\nEND_OF_REPORT\n", parm->report);
		fclose(parm->report);
		parm->report = NULL;
		set_log_file(stderr);
	}
}

static void on_sigusr1(fl_parm* fl)
// reload configuraition file
{
	assert(fl);

	av_parm *parm = get_parm(fl);

	assert(parm && parm->config_file);

	int close_pipe = 0, rtc;

	if (parm && parm->report == NULL)
	{
		open_report_pipe(fl);
		close_pipe = 1;
	}

	FILE *logfile = set_log_file(NULL);

	parm_t b;
	memset(&b, 0, sizeof b);
	if ((rtc = read_all_values(&b, parm->config_file)) == 0)
	{
		apply_parm_defaults(&b);
		if (check_consistency(&b, create_dir, logfile, b.verbose) == 0)
		{
			clear_parm(&parm->a);
			parm->a = b;
			fl_set_verbose(fl, b.verbose);
			set_log_verbose(b.verbose);

			if (close_pipe)
				// not going to create a new engine, so set values now
				// this can yield some errors, such as setting bytecode mode
				// after the engine was compiled...  ignored
				set_engine_values(&b, parm->clamav);
		}
		else
		{
			clear_parm(&b);
			rtc = 1;
		}
	}

	filelog(logfile, LOG_INFO, "New configuration %s",
		rtc? "rejected": "accepted");

	if (close_pipe)
		close_report_pipe(parm);
}

static void on_sigusr2(fl_parm* fl)
// reload virus database data
{
	av_parm *parm = get_parm(fl);
	if (parm->report == NULL)
		open_report_pipe(fl);
	clamav_load_n_report(parm);
	close_report_pipe(parm);
}

static void on_sighup(fl_parm *fl)
// reload both config and data
{
	open_report_pipe(fl);
	on_sigusr1(fl);
	on_sigusr2(fl); // closes report pipe
}

static int report_config(av_parm *parm)
{
	printf("%s\n", parm->config_file);
	print_parm(&parm->a);
	return check_consistency(&parm->a, dont_create_dir, stdout, 1);
}

static void write_pid_file(fl_parm *fl)
{
#if !defined(TEST_AVFILTER) // no pid file for tavfilter
	char path[PATH_MAX];
	av_parm const *parm = fl_get_parm(fl);
	parm_t const *a = parm? &parm->a: NULL; 
	strcpy(path, a && a->piddir? a->piddir: AVFILTER_PID_DIR);
	strcat(path, "/avfilter.pid");
	FILE *fp = fopen(path, "w");
	char const *failed_action = NULL;
	if (fp)
	{
		fprintf(fp, "%lu\n", (unsigned long) getpid());
		if ((ferror(fp) | fclose(fp)) != 0)
			failed_action = "write";
		get_parm(fl)->pid_created = 1;
	}
	else
		failed_action = "open";
	if (failed_action)
	{
		filelog(stderr, LOG_ALERT, "cannot %s %s: %s",
			failed_action, AVFILTER_PID_FILE, strerror(errno));
		filelog(stderr, LOG_ALERT,
			"%s must contain %lu for signalling virus updates",
			AVFILTER_PID_FILE, (unsigned long) getpid());
	}
#else
	(void)fl;
#endif
}

static void delete_pid_file(av_parm *parm)
{
#if !defined(TEST_AVFILTER) // no pid file for tavfilter
	if (parm->pid_created)
	{
		char path[PATH_MAX];
		parm_t const *a = parm? &parm->a: NULL; 
		strcpy(path, a && a->piddir? a->piddir: AVFILTER_PID_DIR);
		strcat(path, "/avfilter.pid");
		if (unlink(path) != 0)
			filelog(stderr, LOG_ERR, "cannot delete %s: %s",
				path, strerror(errno));
	}
#else
	(void)parm;
#endif
}

static inline void libclamav_debug(void)
{
	char *e = getenv("LIBCLAMAV_DEBUG");
	if (e && atoi(e) > 0)
		cl_debug();
}


static fl_init_parm functions =
{
	avfilter,
	write_pid_file,
	on_sighup, on_sigusr1, on_sigusr2,
	NULL, NULL, NULL, NULL
};

static char default_config_file[] = AVFILTER_CONFIG_FILE;
// COURIER_SYSCONF_INSTALL "/filters/avfilter.conf";

static void return_432(fl_parm *fl)
{
	fl_pass_message(fl, "432 Mail filter not loaded.\n");
}

int main(int argc, char *argv[])
{
	int want_config = 0, rtc = 0, alllocal = 0;
	av_parm parm;

	memset(&parm, 0, sizeof parm);
	parm.x55aa5a5a = 0x55aa5a5a;

	for (int i = 1; i < argc; ++i)
	{
		char const *const arg = argv[i];

		if (strcmp(arg, "-f") == 0)
		{
			if (++i < argc)
				parm.config_file = argv[i];
		}
		else if (strcmp(arg, "--version") == 0)
		{
			puts(PACKAGE_STRING);
			return 0;
		}
		else if (strcmp(arg, "--config") == 0)
			want_config |= 1;
		else if (strcmp(arg, "--check-action") == 0)
			want_config |= 2;
		else if (strcmp(arg, "-t") == 0 || strcmp(arg, "--batch") == 0)
			want_config |= 4;
		else if (strcmp(arg, "--all-local") == 0)
			alllocal = 1;
		else if (strcmp(arg, "--help") == 0)
		{
			fputs("avfilter command line args:\n"
			/*  12345678901234567890123456 */
				"  --help                  print this stuff and exit\n"
				"  --version               print version string and exit\n"
				"  -f config-file          specify alternate config file\n"
				"  --config                display values from config file\n"
				"  --check-action          match pass/reject/drop configuration\n",
					stdout);
			fl_main(NULL, NULL, argc - i + 1, argv + i - 1, 0, 0);
			return 0;
		}
	}

	set_log_file(want_config? stdout: stderr);

	if (parm.config_file == NULL)
		parm.config_file = alllocal? "avtest.conf": default_config_file;

	if (read_all_values(&parm.a, parm.config_file))
		rtc = 1;

	if (want_config & 3)
	{
		if (rtc == 0)
		{
			if (getuid() == 0) // Around make install?  Simulate under Courier
			{
				setgid(MAILGID);
				setuid(MAILUID);
			}

			set_log_no_pid(1);
			if (want_config & 1) // --config
			{
				rtc = (want_config & 4) == 0? report_config(&parm):
					check_consistency(&parm.a, dont_create_dir, stdout, 1);
			}
			if (want_config & 2) // --check_action
			{
				check_config_action(&parm);
				rtc = check_consistency(&parm.a, dont_create_dir, stdout, 0);
			}
		}
	}
	else
	{
		apply_parm_defaults(&parm.a);
		set_log_verbose(parm.a.verbose);
		cl_set_clcb_msg(clamav_msg);
		libclamav_debug();
		rtc = cl_init(CL_INIT_DEFAULT);
		if (rtc != CL_SUCCESS)
		{
			filelog(stderr, LOG_ALERT, "cannot initialize libclamav: %s",
				strerror(rtc));
			rtc = 1;
		}
		else if (
			check_consistency(&parm.a, create_dir, stderr, parm.a.verbose) ||
			(rtc = clamav_load_n_report(&parm)) != 0)
		{
			static_assert(CL_SUCCESS == 0);
			if (rtc == CL_SUCCESS) // then check_consistency failed
				filelog(stderr, LOG_ALERT, "inconsistent config in %s",
					parm.config_file);
			rtc = 1;
		}

		if (rtc)
			functions.filter_fn = &return_432;

		if (rtc == 0 || argc == 1)
		{
			rtc = fl_main(&functions, &parm, argc, argv,
				parm.a.all_mode, parm.a.verbose);
		}

		delete_pid_file(&parm);
	}

#if defined DO_CL_ENGINE_FREE
	if (parm.clamav)
		my_cl_engine_free(parm.a.verbose, parm.clamav, "main");
#endif
	clear_parm(&parm.a);

	return rtc;
}
