/*
** avfilter_sig.c - written by vesely in milano on 22jan2004
** utility to send a signal to avfilter if it is running

Copyright (C) 2004-2025 Alessandro Vesely

This file is part of avfilter

avfilter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

avfilter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License version 3
along with avfilter.  If not, see <http://www.gnu.org/licenses/>.


*/

#if defined(HAVE_CONFIG_H)
#include <config.h>
#else
#error config.h missing
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <syslog.h>
#include <limits.h>
#include "log.h"
#include "avfiledefs.h"
#include "parm.h"

static const int default_timeout = 90;

char const syntax_string1[] =
	"    avfilter_sig --help | --version";
char const syntax_string2[] =
	"    avfilter_sig [-l] [-t timeout]  [-p pid] [reload [config | virus]]";

// format string - expect args: syntax_strings 1 and 2,
// pidfile, pipefile, configfile
char const help_string[] =
	PACKAGE_STRING "\n"
	"\n"
	"syntax:\n"
	"   %s\n"
	" or\n"
	"   %s\n"
	"\n"
	"In the second form, avfilter_sig sends a signal to avfilter so that the\n"
	"latter reloads its configuration file, the virus data base, or both. The\n"
	"configuration file of avfilter can be overridden for testing in its own\n"
	"command line, otherwise it is configured as\n"
	"\t%s\n"
	"The timeout of %d seconds can be overridden with -t.\n"
	"The process id of avfilter can be overridden with -p, which also changes\n"
	"the pipe name to the current directory; if not using -p, the id is read\n"
	"from avfilter pid file, which, if not overridden in the configuration file\n"
	"of avfilter, is configured as\n"
	"\t%s\n"
	"A named pipe is used to read the response. Existence of the pipe is tested\n"
	"to avoid concurrent requests. When the -p option is used avfilter_sig uses\n"
	"\"%s\" as the pipe name.\n"
	"For regular operations, avfilter_sig uses the configured pipe, which is\n"
	"\t%s\n"
	"\n"
	"The output from avfilter, read through the pipe, is displayed on stdout\n"
	"unless the -l option is given.  In the latter case it is sent to syslog.\n"
	"\n"
	"The signal sent is\n"
	"HUP  -if neither conf nor virus is specified,\n"
	"USR1 -for conf, to reload configuration,\n"
	"USR2 -for virus, to reload the virus data base.\n"
	"\n"
	"Exit status is 0 if ok, 1 for daemon error, higher for other errors\n"
	"including daemon not running, pipe already in use, etc.\n";


static stdlog_t log_fn = &stdlog;

static int signal_timed_out = 0,
	signal_break = 0;

static int keep_running(int timeout)
// timeout: 0 == reset timeout condition
// return: 1 == continue, 0 == stop
{
	if (signal_break > 0)
	{
		(*log_fn)(LOG_WARNING, "caught %s signal\n",
			signal_break == SIGPIPE ? "PIPE" :
			signal_break == SIGINT ? "INT" :
			signal_break == SIGQUIT ? "QUIT" :
			signal_break == SIGTERM ? "TERM" : "unexpected");
		signal_break = -1;
	}

	if (signal_timed_out > 0)
	{
		if (timeout > 0)
		{
			(*log_fn)(LOG_ERR,
				"terminated after timeout [%d seconds]\n",
				timeout);
			signal_timed_out = -1;
		}
		else
			signal_timed_out = timeout;
	}

	return signal_timed_out == 0 && signal_break == 0;
}

static void sig_catcher(int sig)
{
	switch(sig)
	{
		case SIGALRM:
			signal_timed_out = 1;
			break;

		case SIGPIPE:
		case SIGINT:
		case SIGQUIT:
		case SIGTERM:
			signal_break = sig;
			break;
		default:
			break;
	}
}

static void init(void)
{
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);

	act.sa_flags = 0;
	act.sa_handler = sig_catcher;
	sigaction(SIGALRM, &act, NULL);
	sigaction(SIGPIPE, &act, NULL);
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGTERM, &act, NULL);
}

static int kill_file(char const *pipe_fname)
{
	int rtc = 0;
	if (unlink(pipe_fname))
	{
		(*log_fn)(LOG_WARNING, "cannot remove %s: %s\n",
			pipe_fname, strerror(errno));
		rtc = 2;
	}
	return rtc;
}

static int verify_pid(pid_t pid, char const *pid_from)
{
	errno = 0;
	if (kill(pid, 0))
	{
		(*log_fn)(LOG_ERR,
			"cannot verify process %d (read from %s): %s\n",
			(int)pid,
			pid_from,
			strerror(errno));
		return 2;
	}
	return 0;
}

int main(int argc, char *argv[])
{
	int i;
	int rtc = 0, rrtc;
	int sig = 0;
	int log_syslog = 0;
	char buf[1024];

	char const* pipe_fname = AVFILTER_PIPE_FILE;
	char const* pid_option = NULL;
	char const* pid_from = AVFILTER_PID_FILE;

	int timeout = default_timeout;

	char *avfilter_sig = strrchr(argv[0], '/');
	if (avfilter_sig)
		++avfilter_sig;
	else
		avfilter_sig = argv[0];

	init();

	/*
	** read arguments
	*/
	for (i = 1; i < argc; ++i)
	{
		char *arg = argv[i];
		if (strcmp(arg, "--help") == 0)
		{
			char *fname = COURIER_SYSCONF_INSTALL "/filters/avfilter.conf";

			fprintf(stdout,
				help_string,
				syntax_string1,
				syntax_string2,
				fname,
				default_timeout,
				AVFILTER_PID_FILE,
				AVFILTER_PIPE_TEST,
				AVFILTER_PIPE_FILE);
			return 0;
		}

		if (strcmp(arg, "--version") == 0)
		{
			fprintf(stdout, "%s\n", PACKAGE_STRING);
			return 0;
		}

		if (strcmp(arg, "reload") == 0)
			(void)0;
		else if (strcmp(arg, "config") == 0)
			sig = sig ? SIGHUP : SIGUSR1;
		else if (strcmp(arg, "virus") == 0)
			sig = sig ? SIGHUP : SIGUSR2;
		else if (*arg == '-')
		{
			int ch;
			while ((ch = *(unsigned char *)++arg) != 0)
			{
				if (ch == 'p' || ch == 't')
				{
					if (++i >= argc)
					{
						fprintf(stderr,
							"%s: option -%c requires an argument\n",
							avfilter_sig, ch);
						return 2;
					}
					if (ch == 'p')
					{
						pid_option = argv[i];
						pipe_fname = AVFILTER_PIPE_TEST;
						pid_from = "command line option \"-p\"";
					}
					else
					{
						char *t = NULL;
						unsigned long l = strtoul(argv[i], &t, 0);
						if (t == NULL || *t != 0 || l > INT_MAX || l <= 0)
						{
							fprintf(stderr,
								"%s: invalid argument for option -t: %s\n",
								avfilter_sig, argv[i]);
							return 2;
						}
						timeout = (int)l;
					}
				}
				else if (ch == 'l')
				{
					log_syslog = 1;
				}
				else
				{
					arg = NULL;
					break;
				}
			}
		}
		else
			arg = NULL;

		if (arg == NULL)
		{
			fprintf(stderr, "%s: invalid: %s\nSyntax:\n%s\nor\n%s\n",
				argv[0], argv[i], syntax_string1, syntax_string2);
			return 2;
		}
	}

	/*
	** log errors from here on
	*/
	if (log_syslog)
	{
		openlog("avfilter_sig", LOG_PID, LOG_MAIL);
		log_fn = &syslog;
	}
	else
	{
		set_program_name("avfilter_sig");
		log_fn = &stdlog; // stderr
	}


	/*
	** copy pid into buffer, either from file or from argument
	*/
	pid_t pid = 1;
	if (pid_option)
	{
		i = strlen(pid_option);
		if ((unsigned)i > sizeof buf)
			i = sizeof buf;
		memcpy(buf, pid_option, i);
	}
	else
	{
		FILE *fp = NULL;
		char *piddir = read_directory(AVFILTER_CONFIG_FILE, "piddir");
		if (piddir)
		{
			unsigned len = strlen(piddir);
			char path[len + 20];
			strcat(strcpy(path, piddir), "/avfilter.pid");
			fp = fopen(path, "r");
			if (fp == NULL)
				(*log_fn)(LOG_ERR,
					"cannot open %s (from avfilter.conf): %s\n",
					path, strerror(errno));
			else
				pid_from = "<piddir>/avfilter.pid in " AVFILTER_CONFIG_FILE;
			free(piddir);
		}

		if (fp == NULL)
			fp = fopen(AVFILTER_PID_FILE, "r");

		if (fp == NULL)
		{
			if (errno == ENOENT)
				(*log_fn)(LOG_ERR,
					"%s not found (avfilter not running?)\n",
					AVFILTER_PID_FILE);
			else
				(*log_fn)(LOG_ERR, "cannot read %s: %s\n",
					AVFILTER_PID_FILE, strerror(errno));
			return 2;
		}
		i = fread(buf, 1, sizeof buf, fp);
		fclose(fp);
	}


	/*
	** verify pid
	*/
	if (i > 0)
	{
		unsigned s = i;
		char *t;
		unsigned long l;
		if (s >= sizeof buf)
			s = sizeof buf - 1;
		buf[s] = 0;
		l = strtoul(buf, &t, 0);
		if (t == NULL || t == &buf[0] ||
			(*t != 0 && *t != '\n') || l <= 0)
		{
			(*log_fn)(LOG_ERR, "invalid pid \"%s\" from %s\n",
				buf, pid_from);
			i = 0;
		}

		pid = (pid_t)l;
	}
	else
	{
		(*log_fn)(LOG_ERR, "cannot read pid from %s: %s\n",
			pid_from, strerror(errno));
	}
	if (i <= 0 || pid == 1)
		return 2;

	if (verify_pid(pid, pid_from))
		return 2;

	/*
	** create pipe, writable by avfilter
	*/
	uid_t my_euid = geteuid();
	mode_t perms = S_IRUSR | S_IWUSR;

	if (my_euid != MAILUID && my_euid != 0)
		perms |= S_IWGRP;	// hope the group is good...


	for (int retries = 0; retries < 2 && mkfifo(pipe_fname, perms); ++retries)
	// try once or twice
	{
		if (errno == EEXIST)
		{
			struct stat st;

			if (stat(pipe_fname, &st) == 0)
			{
				unsigned long diff = time(NULL) - st.st_mtime;
				mode_t const mode = st.st_mode & S_IFMT;

				/* if it's stale (older than 2 hours) we can remove it... */
				if (diff > 2UL * 3600UL)
				{
					if (unlink(pipe_fname) == 0)
						continue;
				}
				else if (mode == S_IFIFO)
				{
					char *msg = &buf[0];
					if (diff > 3600UL)
					{
						msg += sprintf(msg, "%ldh ", diff/3600UL);
						diff %= 3600UL;
					}
					if (diff > 60UL)
					{
						msg += sprintf(msg, "%ldm ", diff/60);
						diff %= 60UL;
					}
					sprintf(msg, "%lds", diff);

					(*log_fn)(LOG_ERR, "%s already in use [%s old]\n",
						pipe_fname, buf);
				}

				if (mode != S_IFIFO) // please delete it for me
				{
					(*log_fn)(LOG_ERR, "%s already in use as %s\n",
						pipe_fname,
						mode == S_IFCHR ? "character special" :
						mode == S_IFDIR ? "directory" :
						mode == S_IFBLK ? "block special" :
						mode == S_IFREG ? "regular file" :
							"other file type");
				}
			}
			else
			{
				(*log_fn)(LOG_ERR,
					"avfilter_sig:\n"
					"this shouldn't happen: cannot create pipe because already\n"
					"exists %s\n"
					"however stat reports it's not found... race condition?\n"
					"Please report this bug to " PACKAGE_BUGREPORT "\n",
						pipe_fname);
			}
		}
		else
		{
			(*log_fn)(LOG_ERR, "cannot mkfifo %s: %s\n",
				pipe_fname, strerror(errno));
		}

		return 2;
		break;
	} // end of mkfifo retries

	/*
	** if not running with courier ids, change the group owner or permissions
	** to make sure avfilter can write to the pipe
	*/
	if (my_euid != MAILUID &&
		chown(pipe_fname, MAILUID, MAILGID) &&
		chown(pipe_fname, -1, MAILGID))
	{
		int const save_errno = errno;
		if (chmod(pipe_fname, perms |= S_IWOTH))
			(*log_fn)(LOG_ERR,
				"failed to change group ownership of pipe\n"
				"%s (reason: %s)\n"
				"cannot even make it world writable: %s\n",
					pipe_fname, strerror(save_errno), strerror(errno));
	}


	/*
	** file must be opened for reading before we send a signal
	** to avfilter, otherwise it may get an ENXIO error when
	** trying to open it - for that we'll check if pipe EOF
	** occurs before avfilter had a chance to open or after
	** having read any data.
	*/
	rrtc = 0;
	errno = 0;
	int fd = open(pipe_fname, O_RDONLY
#if defined(O_NONBLOCK)
		| O_NONBLOCK
#elif defined(O_NDELAY)
		| O_NDELAY
#else
#error neither O_NONBLOCK nor O_NDELAY defined
#endif
	);
	if (fd < 0)
	{
		(*log_fn)(LOG_ERR, "cannot open %s for reading: %s\n",
			pipe_fname, strerror(errno));
		rrtc = 2;
	}

	/* Take out of NDELAY mode */
	if (rrtc == 0)
	{
		errno = 0;
		if (fcntl(fd, F_SETFL, 0) == -1)
		{
			(*log_fn)(LOG_ERR,
				"cannot reset file descriptor flags: %s\n",
				strerror(errno));
			close(fd);
			rrtc = 2;
		}
	}

	if (rrtc == 0)
	{
		if (sig == 0)
			sig = SIGHUP;
		errno = 0;
		if (keep_running(0))
		{
			if (kill(pid, sig))
			{
				(*log_fn)(LOG_ERR,
					"cannot signal to process %d (read from %s): %s\n",
					(int)pid,
					pid_from,
					strerror(errno));
				close(fd);
				rrtc = 2;
			}
		}
		else
			rrtc = 2;
	}

	if (rrtc == 0)
	{
		char *const ebuf = &buf[sizeof buf - 1];
		char *sbuf = &buf[0];
		int read_any = 0, read_complete = 0, info_given = 0;

		alarm(timeout);
		while (keep_running(timeout))
		/*
		** read one char at a time until timeout, error or empty line
		*/
		{
			int p;

			errno = 0;
			p = read(fd, sbuf, 1);
			if (p < 0)
			{
				switch (errno)
				{
					case EAGAIN:
					case EINTR:
						if (verify_pid(pid, pid_from) == 0)
							continue;
						/* else thru */
					default:
						break;
				}
				(*log_fn)(LOG_ERR, "cannot read %s: %s\n",
					pipe_fname,
					strerror(errno));
				rrtc = 2;
				break;
			}
			else if (p == 0)
			/*
			** if nothing was read thus far, wait a second and verify the pid
			** again.  cannot intermix sleep and alarm, so use pause.
			** if timeout had expired or pid had gone, break out.
			*/
			{
				if (!read_any)
				{
					int const residual_timeout = alarm(1) - 1;
					pause();
					if (keep_running(residual_timeout > 0 ? 0 : timeout) &&
						verify_pid(pid, pid_from) == 0)
					{
						alarm(residual_timeout);
						if (info_given == 0 && timeout - residual_timeout >= 2)
						{
							static const char info_fmt[] =
								"avfilter not responding, waiting %d seconds%s";
							info_given = 1;
							if (isatty(fileno(stdout)))
								printf(info_fmt, residual_timeout, "\n");
							(*log_fn)(LOG_INFO, info_fmt, residual_timeout, "");
						}
						continue;
					}
				}
				else
					(*log_fn)(LOG_ERR, "unexpected EOF on %s\n", pipe_fname);
				rrtc = 2;
				break;
			}

			read_any = 1;

			/*
			** if one line was read, put it through
			*/
			p = *sbuf++;
			if (p == '\n' || sbuf >= ebuf)
			{
				char *bbuf = &buf[0];
				size_t sz = sbuf - bbuf - 1, szl;
				if (sz > 0) // skip empty lines
				{
					int severity = LOG_ERR;
					*sbuf = 0;
					if (is_severity_encoded(bbuf, &severity, &szl))
					{
						bbuf += szl;
						while (bbuf < sbuf && *bbuf == ' ')
							++bbuf;
						rtc = 1;
					}                            //     1234567890123
					else if (sz >= 13U && memcmp(bbuf, "END_OF_REPORT", 13U) == 0)
					{
						read_complete = 1;
						break;
					}

					if (log_syslog)
						(*log_fn)(severity, "%s", bbuf);
					else
						fputs(bbuf, stdout);
				}
				sbuf = &buf[0];
			}
		} // end of read loop

		if (!read_any)
		{
			(*log_fn)(LOG_ERR, "no data read from pipe\n");
			rrtc = 2;
		}
		else if (!read_complete)
		{
			(*log_fn)(LOG_ERR, "read was not complete\n");
			rrtc = 2;
		}

		if (close(fd))
		{
			(*log_fn)(LOG_ERR, "cannot close %s: %s\n",
				pipe_fname, strerror(errno));
			rrtc = 2;
		}

		alarm(0);
		if (!keep_running(timeout))
			rrtc = 2;
	}

	if (kill_file(pipe_fname))
		rrtc = 2;

	if (rtc == 0)
		rtc = rrtc;

	return rtc;
}
