/*
* log.c - extracted from zdkimfilter/src/parm by ale in milano on 24jun2014
* logging utilities and weariness

Copyright (C) 2014 Alessandro Vesely

This file is part of avfilter

avfilter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

avfilter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License version 3
along with avfilter.  If not, see <http://www.gnu.org/licenses/>.

*/
#include <config.h>
#if !AVFILTER_DEBUG && !NDEBUG
#define NDEBUG
#endif
#include <inttypes.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <sys/types.h>
#include <unistd.h>
#include <limits.h>
#include <ctype.h>
#include <stddef.h>
#include <stdarg.h>
#include "log.h"

#include <clamav.h>

#include <assert.h>


/*
* avfilter logs using fprintf, designed after Courier's error stream.
* other command utilities can get/set the logfun/program_name:
* syslog, and stderrlog are viable candidates.
*/

static const char *program_name = "avfilter";
static FILE* logfile = NULL;
static int verbose;
static int no_pid;

const char* set_program_name(const char * new_name)
{
	const char* rt = program_name;
	if (new_name)
		program_name = new_name;
	return rt;
}

FILE *set_log_file(FILE *fp)
{
	FILE *rt = logfile;
	if (fp)
		logfile = fp;
	else if (rt == NULL)
		logfile = rt = stderr;
	return rt;
}

int set_log_verbose(int new_verbose)
{
	int rt = verbose;
	verbose = new_verbose;
	return rt;
}

int set_log_no_pid(int new_no_pid)
{
	int rt = no_pid;
	no_pid = new_no_pid;
	return rt;
}

#define LOG_NO_NEWLINE 0x800

#if !defined LOG_PRIMASK
#define LOG_PRIMASK \
(LOG_EMERG|LOG_ALERT|LOG_CRIT|LOG_ERR|LOG_WARNING|LOG_NOTICE|LOG_INFO|LOG_DEBUG)
#endif

static void vfilelog(FILE* fp, int severity, char const* fmt, va_list ap)
{
	char const *logmsg;
	switch (severity & LOG_PRIMASK)
	{
		case LOG_EMERG:
		//	logmsg = "EMERG";
		//	break;

		case LOG_ALERT:
			logmsg = "ALERT";
			break;

		case LOG_CRIT:
			logmsg = "CRIT";
			break;

		case LOG_ERR:
		default:
			logmsg = "ERR";
			break;

		case LOG_WARNING:
			logmsg = "WARN";
			break;

		case LOG_NOTICE:
		//	logmsg = "NOTICE";
		//	break;

		case LOG_INFO:
			logmsg = "INFO";
			break;

		case LOG_DEBUG:
			logmsg = "DEBUG";
			break;
	}

	if (fp == stdout)
		fputs(program_name, stdout);
	else
		fprintf(fp, "%s: %s", logmsg, program_name);

	if (no_pid)
		fputs(": ", fp);
	else
		fprintf(fp, "[%d]: ", (int)getpid());

	vfprintf(fp, fmt, ap);
	if ((severity & LOG_NO_NEWLINE) == 0)
		fputc('\n', fp);
}

int is_severity_encoded(char const* buf, int* severity, size_t *length)
{
	char const *code;
	int sev;
	switch (*(unsigned char const*)buf)
	{
		case 'A': code = "ALERT"; sev = LOG_ALERT; break;
		case 'C': code = "CRIT"; sev = LOG_ALERT; break;
		case 'D': code = "DEBUG"; sev = LOG_DEBUG; break;
		case 'E':
			if (buf[1] == 'R')
			{
				code = "ERR"; sev = LOG_ERR; break;
			}

			if (buf[1] == 'M')
			{
				code = "EMERG"; sev = LOG_EMERG; break;
			}

			return 0;

		default:
			return 0;

		case 'I': code = "INFO"; sev = LOG_INFO; break;
		case 'N': code = "NOTICE"; sev = LOG_NOTICE; break;
		case 'W': code = "WARN"; sev = LOG_WARNING; break;
	}
	size_t len = strlen(code);
	if (strncmp(buf, code, len) == 0 && buf[len] == ':')
	{
		if (severity) *severity = sev;
		if (length) *length = len + 1;
		return 1;
	}
	return 0;
}

void filelog(FILE* fp, int severity, char const* fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	vfilelog(fp, severity, fmt, ap);
	va_end(ap);
}

void stdlog(int severity, char const* fmt, ...)
{
	if (logfile == NULL) logfile = stderr;
	va_list ap;
	va_start(ap, fmt);
	vfilelog(logfile, severity, fmt, ap);
	va_end(ap);
}

void
clamav_msg(enum cl_msg severity, const char *nu1, const char *msg, void *nu2)
{
	int sev;
	if (severity >= CL_MSG_ERROR)
		sev = LOG_ERR;
	else if (severity >= CL_MSG_WARN)
	{
		if (verbose <= 0)
			return;
		sev = LOG_WARNING;
	}
	else
	{
		if (verbose <= 1)
			return;
		sev = LOG_INFO;
	}

	stdlog(sev | LOG_NO_NEWLINE, "libClamAV: %s", msg);

	(void)nu1;
	(void)nu2;
}

