/*
** Mimic courierfilter
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <poll.h>
#include <errno.h>
#include <pthread.h>
#include <time.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

typedef struct filterst
{
	pid_t p;
	int fd0;
} filterst;

static int killfilter(filterst *f)
{
	puts("Stopping avfilter");
	close(f->fd0);

	int	waitstat;
	pid_t pid=wait(&waitstat);
	return pid == f->p;
}

/*
**	sighup attempts to synchronize everything.
*/
static void sighup(char const *avfilter, filterst *f)
{
	/* Start avfilter */

	int pipefd[2];
	if (pipe(pipefd) < 0)
	{
		perror("pipe");
		exit(1);
	}

	f->fd0=pipefd[1];
	while ((f->p=fork()) < 0)
	{
		perror("fork");
		sleep(3);
	}

	if (f->p == 0)
	{
		dup2(pipefd[0], 0);
		close(pipefd[0]);
		close(pipefd[1]);

		puts("Starting avfilter");
		struct stat st;
		if (stat(avfilter, &st) == 0 &&
			(st.st_mode & S_IFMT) == S_IFREG &&
			(st.st_mode & S_IXUSR) != 0)
		{
			int sz = strlen(avfilter) + 3;
			char path[sz];
			strncat(strncpy(path, "./", sz), avfilter, sz);
			execl(path, avfilter, "--all-local", (char *)0);
		}
		else
			execlp(avfilter, avfilter, "--all-local", (char *)0);
		perror("exec");
		exit(1);
	}
	close(pipefd[0]);
	if (fcntl(f->fd0, F_SETFD, FD_CLOEXEC) < 0)
	{
		perror("fcntl");
		exit(1);
	}
}

/*
**	Tricky synchronization when we start courierfilter.  Basically, we
**	don't want to exit this process until all filters are initialized.
**
**	Here's what we do.  All filters will manually close file descritor
**	3 when they're ready.  Therefore, we set up a pipe whose write end
**	is set to file descriptor 3, and is inherited via fork and exec
**	by every filter.  We do a read on the pipe, which will complete
**	when all copies of the write side of the pipe are closed.
**
*/
static int reserve3(int* pipe3fd)
{
	int devnull = -1, dupped = -1;
	pipe3fd[0] = pipe3fd[1] = -1;
	close(3);
	if (open("/dev/null", O_RDONLY) != 3
	    || pipe(pipe3fd) < 0
	    || close(3)
	    || (dupped = dup(pipe3fd[1])) != 3)
	{
		fprintf(stderr, "Unable to reserve file descriptor 3.\n");
		close(devnull);
		close(dupped);
		close(pipe3fd[0]);
		close(pipe3fd[1]);
		return (1);
	}
	close(pipe3fd[1]);
	return (0);
}

static int start(char const *avfilter, filterst *f)
{
	int	pipe3fd[2];

	if (reserve3(pipe3fd))
	{
		return (1);
	}

	fcntl(pipe3fd[0], F_SETFD, FD_CLOEXEC);

	signal(SIGPIPE, SIG_IGN);
	dup2(2, 1);
	sighup(avfilter, f);
	close(0);
	open("/dev/null", O_RDONLY);
	close(3);

	char temp_buf;
	if (read(pipe3fd[0], &temp_buf, 1) != 1)
	{
		; /* ignore */
	}
	close(pipe3fd[0]);
	return (0);
}

static char *read_response(int sfd)
{
	unsigned bufsize = 256;
	unsigned start = 0, startline = 0;
	char *repl = malloc(bufsize);
	if (repl == NULL)
	{
		perror("malloc");
		abort();
	}

	errno = 0;
	repl[0] = 0;
	int malformed = 0, complete = 0, try = 0;
	do
	{
		struct pollfd pfd = {0};
		pfd.fd = sfd;
		pfd.events = POLLIN;

		int ready;
		while ((ready = poll(&pfd, 1, 2000)) < 0)
			if (errno != EINTR && errno != EAGAIN)
				break;

		++try;
		if (ready == 1 && (pfd.revents & POLLIN))
		{
			unsigned available;
	#if defined SO_NREAD
			socklen_t optlen = sizeof(available);
			int err = getsockopt(sfd, SOL_SOCKET, SO_NREAD, &available, &optlen);
	#else
			int err = ioctl(sfd, FIONREAD, &available);
	#endif
			if (err == 0)
			{
				unsigned nread = bufsize - start;
				if (available + 1 >= nread)
				{
					bufsize += available + 1;
					char *new_buf = realloc(repl, bufsize);
					if (new_buf == NULL)
					{
						free(repl);
						perror("realloc");
						abort();
					}
					repl = new_buf;
				}
				nread = available;

				ssize_t n = read(sfd, repl + start, nread);
				if (n == nread)
				{
					unsigned stop = start + nread;
					for (unsigned i = start; i < stop; ++i)
						if (repl[i] == '\n')
						{
							if (i - startline >= 3)
							{
								if (repl[startline + 3] == '-')
									startline = i + 1;
								else
								{
									if (i + 1 < stop)
										malformed = 1;
									complete = 1;
								}
							}
							else
								malformed = 1;
						}

					repl[stop] = 0;
					start = stop;
				}
				else perror("read");
			}
			else perror("ioctl");
		}
		else if (ready || errno)
		{
			fprintf(stderr, "poll returned %d errno=%s (%s)\n",
				ready, strerrorname_np(errno), strerror(errno));
		}
	} while (malformed == 0 && complete == 0 && errno == 0 && try < 4);

	if (start > 0)
		repl[start - 1] = '\n';
	return repl;
}

static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
static char const ctl[] = "ctl";
static char **argv_files;
static int print_time = 0;
static struct sockaddr_un addr;
typedef struct thread_arg
{
	pthread_t id;
	int argc;
} thread_arg;

void *run_file(void *varg)
{
	thread_arg *arg = varg;
	int ndx = arg->argc;
	int sfd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (sfd >= 0)
	{
		if (connect(sfd, (struct sockaddr *) &addr,
			sizeof(struct sockaddr_un)) == 0)
		{
			struct timespec before, after;
			FILE *f_send = fdopen(dup(sfd), "wb");
			int rtc = clock_gettime(CLOCK_MONOTONIC, &before);
			fprintf(f_send, "%s\n%s\n\n", argv_files[ndx], ctl);
			fclose(f_send);
			char *repl = read_response(sfd);
			rtc |= clock_gettime(CLOCK_MONOTONIC, &after);
			if ((errno = pthread_mutex_lock(&mutex)) != 0)
				perror("pthread_mutex_lock");
			else
			{
				if (print_time)
					printf("%3d: %s\n", ndx, argv_files[ndx]);
				if (repl[0])
					fputs(repl, stdout);
				if (rtc == 0 && print_time)
				{
					struct timeval b, a, diff;
					TIMESPEC_TO_TIMEVAL(&a, &after);
					TIMESPEC_TO_TIMEVAL(&b, &before);
					timersub(&a, &b, &diff);
					printf("total time %ld.%06lds\n",
						diff.tv_sec, diff.tv_usec);
				}
				if ((errno = pthread_mutex_unlock(&mutex)))
					perror("pthread_mutex_unlock");
			}
			free(repl);
		}
		else perror("connect socket");
	}
	close(sfd);
	return NULL;
}

int main(int argc, char *argv[])
{
	int i, rtc = 1;
	for (i = 1; i < argc; ++i)
	{
		if (strcmp(argv[i], "--help") == 0)
		{
			printf("Usage:\n"
	"\t%s [opt] [mailfile...]\n"
	"%s runs and stops avfilter, mimicking courierfilter.  Any mailfile is\n"
	"passed to avfilter along with a dummy control file.\n"
	"The only supported option is -t, to print the timing each file took,\n"
	"from connecting the socket to full line output\n"
	"%s creates files named \"s_avfilter\" and \"ctl\" in the current directory\n",
	argv[0], argv[0], argv[0]);
			return 0;
		}

		if (strcmp(argv[i], "-t") == 0)
			print_time = 1;
		else if (strcmp(argv[i], "--") == 0)
			break;
		else if (argv[i][0] != '-')
			break;
		else
		{
			fprintf(stderr, "unknown option %s\n", argv[i]);
			return 1;
		}
	}

	argv_files = argv + i;
	int argc_files = argc - i;

	FILE *fp = fopen(ctl, "w");
	if (fp)
	{
		fputs("ranyone@example.com\n", fp);
		fclose(fp);
		filterst f;
		memset(&f, 0, sizeof f);
		if (start("avfilter", &f) == 0)
		{
			if (argc > 2)
			{
				addr.sun_family = AF_UNIX;
				strncpy(addr.sun_path, "./s_avfilter", sizeof(addr.sun_path) - 1);

				thread_arg *subs = calloc(argc_files, sizeof(thread_arg));
				if (subs == NULL)
					abort();
				for (i = 0; i < argc_files; ++i)
				{
					subs[i].argc = i;
					if ((errno = pthread_create(&subs[i].id,
						NULL, &run_file, &subs[i])) != 0)
					{
						perror("pthread_create");
						argc_files = i;
						break;
					}
				}
				void *res;
				for (i = 0; i < argc_files; ++i)
					if ((errno = pthread_join(subs[i].id, &res)) != 0)
						perror("pthread_join");
				free(subs);
			}
			rtc = killfilter(&f);
		}
		unlink(ctl);
	}
	else perror("fopen");
	return rtc;
}
